<?php

namespace App\Http\Controllers;

use App\Models\AccountPayable;
use App\Models\Supplier;
use App\Models\Category;
use App\Http\Requests\AccountPayableRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class AccountPayableController extends Controller
{
    public function index(Request $request)
    {
        $query = AccountPayable::where('company_id', Auth::user()->company_id)
            ->with(['supplier', 'category']);

        // Filtros
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('due_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('due_date', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $accounts = $query->orderBy('due_date', 'desc')->paginate(20);

        // Dados para filtros
        $suppliers = Supplier::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'debit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        // Estatísticas
        $stats = [
            'total_pending' => AccountPayable::where('company_id', Auth::user()->company_id)
                ->where('status', 'pending')
                ->sum('amount'),
            'total_overdue' => AccountPayable::where('company_id', Auth::user()->company_id)
                ->where('status', 'pending')
                ->where('due_date', '<', now())
                ->sum('amount'),
            'total_paid_month' => AccountPayable::where('company_id', Auth::user()->company_id)
                ->where('status', 'paid')
                ->whereMonth('paid_date', now()->month)
                ->sum('amount'),
        ];

        return view('accounts-payable.index', compact('accounts', 'suppliers', 'categories', 'stats'));
    }

    public function create()
    {
        $suppliers = Supplier::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'debit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        return view('accounts-payable.create', compact('suppliers', 'categories'));
    }

    public function store(AccountPayableRequest $request)
    {
        AccountPayable::create([
            ...$request->validated(),
            'company_id' => Auth::user()->company_id,
        ]);

        return redirect()->route('accounts-payable.index')
            ->with('success', 'Conta a pagar cadastrada com sucesso!');
    }

    public function show(AccountPayable $accountPayable)
    {
        $this->authorize('view', $accountPayable);
        
        $accountPayable->load(['supplier', 'category']);
        
        return view('accounts-payable.show', compact('accountPayable'));
    }

    public function edit(AccountPayable $accountPayable)
    {
        $this->authorize('update', $accountPayable);
        
        $suppliers = Supplier::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'debit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        return view('accounts-payable.edit', compact('accountPayable', 'suppliers', 'categories'));
    }

    public function update(AccountPayableRequest $request, AccountPayable $accountPayable)
    {
        $this->authorize('update', $accountPayable);
        
        $accountPayable->update($request->validated());

        return redirect()->route('accounts-payable.show', $accountPayable)
            ->with('success', 'Conta a pagar atualizada com sucesso!');
    }

    public function destroy(AccountPayable $accountPayable)
    {
        $this->authorize('delete', $accountPayable);
        
        if ($accountPayable->status === 'paid') {
            return back()->with('error', 'Não é possível excluir conta já paga.');
        }

        $accountPayable->delete();

        return redirect()->route('accounts-payable.index')
            ->with('success', 'Conta a pagar excluída com sucesso!');
    }

    public function pay(Request $request, AccountPayable $accountPayable)
    {
        $this->authorize('update', $accountPayable);
        
        $request->validate([
            'paid_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        $accountPayable->update([
            'status' => 'paid',
            'paid_date' => $request->paid_date,
            'notes' => $request->notes,
        ]);

        return back()->with('success', 'Pagamento registrado com sucesso!');
    }
}
