<?php

namespace App\Http\Controllers;

use App\Models\AccountReceivable;
use App\Models\Customer;
use App\Models\Category;
use App\Http\Requests\AccountReceivableRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AccountReceivableController extends Controller
{
    public function index(Request $request)
    {
        $query = AccountReceivable::where('company_id', Auth::user()->company_id)
            ->with(['customer', 'category']);

        // Filtros
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('due_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('due_date', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $accounts = $query->orderBy('due_date', 'desc')->paginate(20);

        // Dados para filtros
        $customers = Customer::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'credit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        // Estatísticas
        $stats = [
            'total_pending' => AccountReceivable::where('company_id', Auth::user()->company_id)
                ->where('status', 'pending')
                ->sum('amount'),
            'total_overdue' => AccountReceivable::where('company_id', Auth::user()->company_id)
                ->where('status', 'pending')
                ->where('due_date', '<', now())
                ->sum('amount'),
            'total_received_month' => AccountReceivable::where('company_id', Auth::user()->company_id)
                ->where('status', 'received')
                ->whereMonth('received_date', now()->month)
                ->sum('amount'),
        ];

        return view('accounts-receivable.index', compact('accounts', 'customers', 'categories', 'stats'));
    }

    public function create()
    {
        $customers = Customer::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'credit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        return view('accounts-receivable.create', compact('customers', 'categories'));
    }

    public function store(AccountReceivableRequest $request)
    {
        DB::transaction(function() use ($request) {
            $account = AccountReceivable::create([
                ...$request->validated(),
                'company_id' => Auth::user()->company_id,
            ]);

            // Se marcado para lançar na conta corrente do cliente
            if ($request->boolean('add_to_customer_account') && $request->customer_id) {
                $customer = Customer::find($request->customer_id);
                $customer->entries()->create([
                    'description' => $request->description,
                    'amount' => $request->amount,
                    'type' => 'debit', // Cliente deve para nós
                    'entry_date' => $request->due_date,
                    'category_id' => $request->category_id,
                    'company_id' => Auth::user()->company_id,
                ]);
                $customer->updateBalance();
            }
        });

        return redirect()->route('accounts-receivable.index')
            ->with('success', 'Conta a receber cadastrada com sucesso!');
    }

    public function show(AccountReceivable $accountReceivable)
    {
        $this->authorize('view', $accountReceivable);
        
        $accountReceivable->load(['customer', 'category']);
        
        return view('accounts-receivable.show', compact('accountReceivable'));
    }

    public function receive(Request $request, AccountReceivable $accountReceivable)
    {
        $this->authorize('update', $accountReceivable);
        
        $request->validate([
            'received_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function() use ($request, $accountReceivable) {
            $accountReceivable->update([
                'status' => 'received',
                'received_date' => $request->received_date,
                'notes' => $request->notes,
            ]);

            // Atualizar saldo do cliente se houver
            if ($accountReceivable->customer) {
                $accountReceivable->customer->entries()->create([
                    'description' => 'Pagamento: ' . $accountReceivable->description,
                    'amount' => $accountReceivable->amount,
                    'type' => 'credit', // Cliente pagou
                    'entry_date' => $request->received_date,
                    'category_id' => $accountReceivable->category_id,
                    'company_id' => Auth::user()->company_id,
                ]);
                $accountReceivable->customer->updateBalance();
            }
        });

        return back()->with('success', 'Recebimento registrado com sucesso!');
    }
}
