<?php

namespace App\Http\Controllers;

use App\Models\BankReconciliation;
use App\Models\BankAccount;
use App\Models\CreditCard;
use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use App\Models\Category;
use App\Services\BankImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class BankReconciliationController extends Controller
{
    protected $bankImportService;

    public function __construct(BankImportService $bankImportService)
    {
        $this->bankImportService = $bankImportService;
    }

    public function index(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        $query = BankReconciliation::where('company_id', $companyId)
            ->with(['bankAccount', 'accountPayable', 'accountReceivable', 'category']);

        // Filtros
        if ($request->filled('bank_account_id')) {
            $query->where('bank_account_id', $request->bank_account_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('transaction_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('transaction_date', '<=', $request->date_to);
        }

        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $reconciliations = $query->orderBy('transaction_date', 'desc')->paginate(20);

        // Dados para filtros
        $bankAccounts = BankAccount::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $creditCards = CreditCard::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        // Estatísticas
        $stats = [
            'total_pending' => BankReconciliation::where('company_id', $companyId)
                ->where('status', 'pending')
                ->count(),
            'total_reconciled' => BankReconciliation::where('company_id', $companyId)
                ->where('status', 'reconciled')
                ->count(),
            'pending_amount' => BankReconciliation::where('company_id', $companyId)
                ->where('status', 'pending')
                ->sum('amount'),
        ];

        return view('bank-reconciliation.index', compact(
            'reconciliations',
            'bankAccounts',
            'creditCards',
            'stats'
        ));
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:pdf,csv,xls,xlsx,txt|max:10240',
            'bank_account_id' => 'required_without:credit_card_id|exists:bank_accounts,id',
            'credit_card_id' => 'required_without:bank_account_id|exists:credit_cards,id',
            'file_type' => 'required|in:pdf,csv,excel,txt',
            'bank_name' => 'required|string',
        ]);

        try {
            $file = $request->file('file');
            $filePath = $file->store('bank_imports', 'local');
            
            $result = $this->bankImportService->importFile(
                $filePath,
                $request->file_type,
                $request->bank_name,
                $request->bank_account_id,
                $request->credit_card_id,
                Auth::user()->company_id
            );

            return back()->with('success', 
                "Importação concluída! {$result['imported']} transações importadas, {$result['duplicates']} duplicatas ignoradas."
            );

        } catch (\Exception $e) {
            return back()->with('error', 'Erro na importação: ' . $e->getMessage());
        }
    }

    public function reconcile(Request $request, BankReconciliation $reconciliation)
    {
        $this->authorize('update', $reconciliation);
        
        $request->validate([
            'action' => 'required|in:reconcile,ignore,link_payable,link_receivable',
            'account_payable_id' => 'nullable|exists:accounts_payable,id',
            'account_receivable_id' => 'nullable|exists:accounts_receivable,id',
            'category_id' => 'nullable|exists:categories,id',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function() use ($request, $reconciliation) {
            switch ($request->action) {
                case 'reconcile':
                    $reconciliation->update([
                        'status' => 'reconciled',
                        'category_id' => $request->category_id,
                        'notes' => $request->notes,
                    ]);
                    break;

                case 'ignore':
                    $reconciliation->update([
                        'status' => 'ignored',
                        'notes' => $request->notes,
                    ]);
                    break;

                case 'link_payable':
                    if ($request->account_payable_id) {
                        $accountPayable = AccountPayable::find($request->account_payable_id);
                        $accountPayable->update([
                            'status' => 'paid',
                            'paid_date' => $reconciliation->transaction_date,
                        ]);
                        
                        $reconciliation->update([
                            'status' => 'reconciled',
                            'account_payable_id' => $request->account_payable_id,
                            'notes' => $request->notes,
                        ]);
                    }
                    break;

                case 'link_receivable':
                    if ($request->account_receivable_id) {
                        $accountReceivable = AccountReceivable::find($request->account_receivable_id);
                        $accountReceivable->update([
                            'status' => 'received',
                            'received_date' => $reconciliation->transaction_date,
                        ]);
                        
                        $reconciliation->update([
                            'status' => 'reconciled',
                            'account_receivable_id' => $request->account_receivable_id,
                            'notes' => $request->notes,
                        ]);
                    }
                    break;
            }

            // Atualizar saldo da conta bancária
            $this->updateBankAccountBalance($reconciliation->bank_account_id);
        });

        return back()->with('success', 'Transação processada com sucesso!');
    }

    public function autoReconcile(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $bankAccountId = $request->bank_account_id;

        $reconciled = $this->bankImportService->autoReconcile($companyId, $bankAccountId);

        return back()->with('success', "{$reconciled} transações foram conciliadas automaticamente.");
    }

    private function updateBankAccountBalance($bankAccountId)
    {
        $bankAccount = BankAccount::find($bankAccountId);
        
        $reconciledCredits = BankReconciliation::where('bank_account_id', $bankAccountId)
            ->where('status', 'reconciled')
            ->where('type', 'credit')
            ->sum('amount');

        $reconciledDebits = BankReconciliation::where('bank_account_id', $bankAccountId)
            ->where('status', 'reconciled')
            ->where('type', 'debit')
            ->sum('amount');

        $newBalance = $bankAccount->initial_balance + $reconciledCredits - $reconciledDebits;
        
        $bankAccount->update(['current_balance' => $newBalance]);
    }
}
