<?php

namespace App\Http\Controllers;

use App\Models\Car;
use App\Models\CarExpense;
use App\Models\Category;
use App\Http\Requests\CarRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class CarController extends Controller
{
    public function index(Request $request)
    {
        $query = Car::where('company_id', Auth::user()->company_id)
            ->withSum('expenses', 'amount');

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('model', 'like', "%{$search}%")
                  ->orWhere('brand', 'like', "%{$search}%")
                  ->orWhere('plate', 'like', "%{$search}%");
            });
        }

        $cars = $query->orderBy('created_at', 'desc')->paginate(12);

        // Estatísticas
        $stats = [
            'total_cars' => Car::where('company_id', Auth::user()->company_id)->count(),
            'cars_owned' => Car::where('company_id', Auth::user()->company_id)->where('status', 'owned')->count(),
            'cars_sold' => Car::where('company_id', Auth::user()->company_id)->where('status', 'sold')->count(),
            'total_invested' => Car::where('company_id', Auth::user()->company_id)->sum('purchase_price'),
            'total_expenses' => CarExpense::whereHas('car', function($q) {
                $q->where('company_id', Auth::user()->company_id);
            })->sum('amount'),
        ];

        return view('cars.index', compact('cars', 'stats'));
    }

    public function create()
    {
        return view('cars.create');
    }

    public function store(CarRequest $request)
    {
        $data = $request->validated();
        $data['company_id'] = Auth::user()->company_id;

        // Upload de fotos
        if ($request->hasFile('photos')) {
            $photos = [];
            foreach ($request->file('photos') as $photo) {
                $path = $photo->store('cars', 'public');
                $photos[] = $path;
            }
            $data['photos'] = $photos;
        }

        Car::create($data);

        return redirect()->route('cars.index')
            ->with('success', 'Veículo cadastrado com sucesso!');
    }

    public function show(Car $car)
    {
        $this->authorize('view', $car);
        
        $car->load(['expenses.category']);
        
        $expenses = $car->expenses()
            ->with('category')
            ->orderBy('date', 'desc')
            ->paginate(15);

        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('type', 'debit')
            ->where('active', true)
            ->orderBy('name')
            ->get();

        // Cálculos
        $totalExpenses = $car->expenses()->sum('amount');
        $profitLoss = 0;
        
        if ($car->sale_price && $car->purchase_price) {
            $profitLoss = $car->sale_price - $car->purchase_price - $totalExpenses;
        }

        return view('cars.show', compact('car', 'expenses', 'categories', 'totalExpenses', 'profitLoss'));
    }

    public function addExpense(Request $request, Car $car)
    {
        $this->authorize('update', $car);
        
        $request->validate([
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0.01',
            'date' => 'required|date',
            'category_id' => 'nullable|exists:categories,id',
        ]);

        CarExpense::create([
            'car_id' => $car->id,
            'description' => $request->description,
            'amount' => $request->amount,
            'date' => $request->date,
            'category_id' => $request->category_id,
        ]);

        return back()->with('success', 'Despesa adicionada com sucesso!');
    }

    public function sell(Request $request, Car $car)
    {
        $this->authorize('update', $car);
        
        $request->validate([
            'sale_price' => 'required|numeric|min:0.01',
            'sale_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        $car->update([
            'sale_price' => $request->sale_price,
            'sale_date' => $request->sale_date,
            'status' => 'sold',
            'notes' => $request->notes,
        ]);

        return back()->with('success', 'Venda registrada com sucesso!');
    }
}
