<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\DefaultCategory;
use App\Http\Requests\CategoryRequest;
use App\Services\CategoryLibraryService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CategoryController extends Controller
{
    protected $categoryLibraryService;

    public function __construct(CategoryLibraryService $categoryLibraryService)
    {
        $this->categoryLibraryService = $categoryLibraryService;
    }

    public function index(Request $request)
    {
        $query = Category::where('company_id', Auth::user()->company_id)
            ->with('defaultCategory');

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        $categories = $query->orderBy('type')->orderBy('name')->paginate(20);
        $statistics = $this->categoryLibraryService->getStatistics();

        return view('categories.index', compact('categories', 'statistics'));
    }

    public function library(Request $request)
    {
        $filters = $request->only(['type', 'group', 'search', 'popular']);
        $categories = $this->categoryLibraryService->getAvailableCategories($filters);
        $groups = $this->categoryLibraryService->getGroups();
        $imported = $this->categoryLibraryService->getImportedCategories();
        $statistics = $this->categoryLibraryService->getStatistics();

        // Agrupar categorias por grupo
        $categoriesByGroup = $categories->groupBy('group');

        return view('categories.library', compact(
            'categoriesByGroup', 
            'groups', 
            'imported', 
            'statistics',
            'filters'
        ));
    }

    public function create()
    {
        return view('categories.create');
    }

    public function store(CategoryRequest $request)
    {
        $data = $request->validated();
        $data['company_id'] = Auth::user()->company_id;
        
        if ($request->filled('keywords_string')) {
            $data['keywords'] = array_map('trim', explode(',', $request->keywords_string));
        }

        Category::create($data);

        return redirect()->route('categories.index')
            ->with('success', 'Categoria criada com sucesso!');
    }

    public function show(Category $category)
    {
        $this->authorize('view', $category);
        
        $usage = [
            'accounts_payable' => $category->accountsPayable()->count(),
            'accounts_receivable' => $category->accountsReceivable()->count(),
        ];

        return view('categories.show', compact('category', 'usage'));
    }

    public function edit(Category $category)
    {
        $this->authorize('update', $category);
        return view('categories.edit', compact('category'));
    }

    public function update(CategoryRequest $request, Category $category)
    {
        $this->authorize('update', $category);

        $data = $request->validated();
        
        if ($request->filled('keywords_string')) {
            $data['keywords'] = array_map('trim', explode(',', $request->keywords_string));
        }

        $category->update($data);

        return redirect()->route('categories.index')
            ->with('success', 'Categoria atualizada com sucesso!');
    }

    public function destroy(Category $category)
    {
        $this->authorize('delete', $category);

        // Verificar se está sendo usada
        $usage = $category->accountsPayable()->count() + $category->accountsReceivable()->count();
        
        if ($usage > 0) {
            return back()->with('error', 'Não é possível excluir uma categoria que está sendo utilizada.');
        }

        $category->delete();

        return redirect()->route('categories.index')
            ->with('success', 'Categoria excluída com sucesso!');
    }

    public function importSelected(Request $request)
    {
        $request->validate([
            'categories' => 'required|array',
            'categories.*' => 'exists:default_categories,id'
        ]);

        $result = $this->categoryLibraryService->importMultiple($request->categories);

        $message = "Importadas {$result['imported']} categorias";
        if ($result['skipped'] > 0) {
            $message .= " ({$result['skipped']} já existiam)";
        }

        return back()->with('success', $message);
    }

    public function importGroup(Request $request)
    {
        $request->validate([
            'group' => 'required|string'
        ]);

        $result = $this->categoryLibraryService->importByGroup($request->group);

        $message = "Importadas {$result['imported']} categorias do grupo '{$request->group}'";
        if ($result['skipped'] > 0) {
            $message .= " ({$result['skipped']} já existiam)";
        }

        return back()->with('success', $message);
    }

    public function importPopular()
    {
        $result = $this->categoryLibraryService->importPopular();

        $message = "Importadas {$result['imported']} categorias populares";
        if ($result['skipped'] > 0) {
            $message .= " ({$result['skipped']} já existiam)";
        }

        return back()->with('success', $message);
    }

    public function initializeDefaults()
    {
        // Importar categorias essenciais
        $essentialCategories = [
            'Combustível', 'Manutenção Veicular', 'Seguro Veicular',
            'Material de Escritório', 'Software', 'Internet',
            'Venda de Veículos', 'Prestação de Serviços'
        ];

        $imported = 0;
        foreach ($essentialCategories as $categoryName) {
            $defaultCategory = DefaultCategory::where('name', $categoryName)->first();
            if ($defaultCategory) {
                $result = $this->categoryLibraryService->importCategory($defaultCategory->id);
                if ($result['success']) {
                    $imported++;
                }
            }
        }

        return back()->with('success', "Inicializadas {$imported} categorias essenciais!");
    }
}
