<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerEntry;
use App\Models\Category;
use App\Http\Requests\CustomerRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        $query = Customer::where('company_id', Auth::user()->company_id)
            ->with(['entries' => function($q) {
                $q->latest()->limit(3);
            }]);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('cpf_cnpj', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('active', $request->status === 'active');
        }

        $customers = $query->paginate(15);

        return view('customers.index', compact('customers'));
    }

    public function create()
    {
        return view('customers.create');
    }

    public function store(CustomerRequest $request)
    {
        $customer = Customer::create([
            ...$request->validated(),
            'company_id' => Auth::user()->company_id,
        ]);

        return redirect()->route('customers.show', $customer)
            ->with('success', 'Cliente cadastrado com sucesso!');
    }

    public function show(Customer $customer)
    {
        $this->authorize('view', $customer);
        
        $customer->load(['entries.category']);
        
        $entries = $customer->entries()
            ->with('category')
            ->orderBy('entry_date', 'desc')
            ->paginate(20);

        $balance = $customer->getBalanceAttribute();
        
        $categories = Category::where('company_id', Auth::user()->company_id)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        return view('customers.show', compact('customer', 'entries', 'balance', 'categories'));
    }

    public function edit(Customer $customer)
    {
        $this->authorize('update', $customer);
        return view('customers.edit', compact('customer'));
    }

    public function update(CustomerRequest $request, Customer $customer)
    {
        $this->authorize('update', $customer);
        
        $customer->update($request->validated());

        return redirect()->route('customers.show', $customer)
            ->with('success', 'Cliente atualizado com sucesso!');
    }

    public function destroy(Customer $customer)
    {
        $this->authorize('delete', $customer);
        
        if ($customer->entries()->count() > 0) {
            return back()->with('error', 'Não é possível excluir cliente com lançamentos.');
        }

        $customer->delete();

        return redirect()->route('customers.index')
            ->with('success', 'Cliente excluído com sucesso!');
    }

    public function addEntry(Request $request, Customer $customer)
    {
        $this->authorize('update', $customer);
        
        $request->validate([
            'description' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:debit,credit',
            'entry_date' => 'required|date',
            'category_id' => 'nullable|exists:categories,id',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function() use ($request, $customer) {
            CustomerEntry::create([
                'customer_id' => $customer->id,
                'company_id' => Auth::user()->company_id,
                'description' => $request->description,
                'amount' => $request->amount,
                'type' => $request->type,
                'entry_date' => $request->entry_date,
                'category_id' => $request->category_id,
                'notes' => $request->notes,
            ]);

            $customer->updateBalance();
        });

        return back()->with('success', 'Lançamento adicionado com sucesso!');
    }

    public function statement(Customer $customer)
    {
        $this->authorize('view', $customer);
        
        $entries = $customer->entries()
            ->with('category')
            ->orderBy('entry_date', 'desc')
            ->get();

        return view('customers.statement', compact('customer', 'entries'));
    }
}
