<?php

namespace App\Http\Controllers;

use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use App\Models\Customer;
use App\Models\Car;
use App\Models\Category;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $companyId = Auth::user()->company_id;
        
        // Estatísticas básicas
        $stats = [
            'total_receivable' => AccountReceivable::where('company_id', $companyId)
                ->where('status', 'pending')
                ->sum('amount'),
            'total_payable' => AccountPayable::where('company_id', $companyId)
                ->where('status', 'pending')
                ->sum('amount'),
            'total_customers' => Customer::where('company_id', $companyId)
                ->where('active', true)
                ->count(),
            'total_cars' => Car::where('company_id', $companyId)->count(),
        ];

        // Próximos vencimentos (próximos 7 dias)
        $upcomingPayments = AccountPayable::where('company_id', $companyId)
            ->where('status', 'pending')
            ->whereBetween('due_date', [now(), now()->addDays(7)])
            ->orderBy('due_date')
            ->limit(5)
            ->get();

        // Contas em atraso
        $overdueAccounts = collect()
            ->merge(
                AccountPayable::where('company_id', $companyId)
                    ->where('status', 'pending')
                    ->where('due_date', '<', now())
                    ->get()
            )
            ->merge(
                AccountReceivable::where('company_id', $companyId)
                    ->where('status', 'pending')
                    ->where('due_date', '<', now())
                    ->get()
            )
            ->sortBy('due_date')
            ->take(5);

        // Top categorias (últimos 30 dias)
        $topCategories = Category::where('company_id', $companyId)
            ->withSum(['accountsPayable as total' => function($query) {
                $query->where('created_at', '>=', now()->subDays(30));
            }], 'amount')
            ->orderByDesc('total')
            ->limit(5)
            ->get();

        // Dados para gráfico (últimos 30 dias)
        $chartData = $this->getChartData($companyId);

        return view('dashboard', compact(
            'stats',
            'upcomingPayments',
            'overdueAccounts',
            'topCategories',
            'chartData'
        ))->with($chartData);
    }

    private function getChartData($companyId)
    {
        $days = collect();
        $income = collect();
        $expenses = collect();

        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $days->push($date->format('d/m'));
            
            $dayIncome = AccountReceivable::where('company_id', $companyId)
                ->whereDate('received_date', $date)
                ->sum('amount');
            
            $dayExpenses = AccountPayable::where('company_id', $companyId)
                ->whereDate('paid_date', $date)
                ->sum('amount');
            
            $income->push($dayIncome);
            $expenses->push($dayExpenses);
        }

        return [
            'chart_labels' => $days->toArray(),
            'chart_income' => $income->toArray(),
            'chart_expenses' => $expenses->toArray(),
        ];
    }
}
