<?php

namespace App\Http\Controllers;

use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Car;
use App\Models\CarExpense;
use App\Services\ReportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ReportController extends Controller
{
    protected $reportService;

    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    public function financial(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $dateFrom = $request->date_from ? Carbon::parse($request->date_from) : now()->startOfMonth();
        $dateTo = $request->date_to ? Carbon::parse($request->date_to) : now()->endOfMonth();

        // Contas a Pagar
        $accountsPayable = AccountPayable::where('company_id', $companyId)
            ->with(['supplier', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();

        // Contas a Receber
        $accountsReceivable = AccountReceivable::where('company_id', $companyId)
            ->with(['customer', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();

        // Resumo
        $summary = [
            'total_payable' => $accountsPayable->sum('amount'),
            'total_payable_paid' => $accountsPayable->where('status', 'paid')->sum('amount'),
            'total_payable_pending' => $accountsPayable->where('status', 'pending')->sum('amount'),
            'total_receivable' => $accountsReceivable->sum('amount'),
            'total_receivable_received' => $accountsReceivable->where('status', 'received')->sum('amount'),
            'total_receivable_pending' => $accountsReceivable->where('status', 'pending')->sum('amount'),
        ];

        $summary['balance'] = $summary['total_receivable_received'] - $summary['total_payable_paid'];

        // Dados para gráfico
        $chartData = $this->getFinancialChartData($companyId, $dateFrom, $dateTo);

        return view('reports.financial', compact(
            'accountsPayable',
            'accountsReceivable',
            'summary',
            'dateFrom',
            'dateTo',
            'chartData'
        ));
    }

    public function customers(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        $customers = Customer::where('company_id', $companyId)
            ->with(['entries', 'accountsReceivable'])
            ->withSum('entries as total_credits', 'amount', function($query) {
                $query->where('type', 'credit');
            })
            ->withSum('entries as total_debits', 'amount', function($query) {
                $query->where('type', 'debit');
            })
            ->get();

        // Calcular saldos
        $customers->each(function($customer) {
            $customer->balance = ($customer->total_credits ?? 0) - ($customer->total_debits ?? 0);
            $customer->pending_receivables = $customer->accountsReceivable()
                ->where('status', 'pending')
                ->sum('amount');
        });

        return view('reports.customers', compact('customers'));
    }

    public function suppliers(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        $suppliers = Supplier::where('company_id', $companyId)
            ->with('accountsPayable')
            ->withSum('accountsPayable as total_payable', 'amount')
            ->withSum('accountsPayable as total_paid', 'amount', function($query) {
                $query->where('status', 'paid');
            })
            ->withSum('accountsPayable as total_pending', 'amount', function($query) {
                $query->where('status', 'pending');
            })
            ->get();

        return view('reports.suppliers', compact('suppliers'));
    }

    public function cars(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        $cars = Car::where('company_id', $companyId)
            ->with('expenses')
            ->withSum('expenses', 'amount')
            ->get();

        // Calcular lucros/prejuízos
        $cars->each(function($car) {
            $car->total_expenses = $car->expenses_sum_amount ?? 0;
            
            if ($car->status === 'sold' && $car->purchase_price && $car->sale_price) {
                $car->profit_loss = $car->sale_price - $car->purchase_price - $car->total_expenses;
            } else {
                $car->profit_loss = 0;
            }
        });

        $summary = [
            'total_cars' => $cars->count(),
            'cars_sold' => $cars->where('status', 'sold')->count(),
            'total_invested' => $cars->sum('purchase_price'),
            'total_sold' => $cars->where('status', 'sold')->sum('sale_price'),
            'total_expenses' => $cars->sum('total_expenses'),
            'total_profit' => $cars->where('profit_loss', '>', 0)->sum('profit_loss'),
            'total_loss' => abs($cars->where('profit_loss', '<', 0)->sum('profit_loss')),
        ];

        return view('reports.cars', compact('cars', 'summary'));
    }

    public function cashFlow(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $dateFrom = $request->date_from ? Carbon::parse($request->date_from) : now()->startOfMonth();
        $dateTo = $request->date_to ? Carbon::parse($request->date_to) : now()->endOfMonth();

        $cashFlow = $this->generateCashFlowData($companyId, $dateFrom, $dateTo);

        return view('reports.cash-flow', compact('cashFlow', 'dateFrom', 'dateTo'));
    }

    // Exportações
    public function exportFinancial(Request $request)
    {
        $format = $request->format ?? 'excel';
        $companyId = Auth::user()->company_id;
        $dateFrom = $request->date_from ? Carbon::parse($request->date_from) : now()->startOfMonth();
        $dateTo = $request->date_to ? Carbon::parse($request->date_to) : now()->endOfMonth();

        if ($format === 'pdf') {
            return $this->reportService->exportFinancialPDF($companyId, $dateFrom, $dateTo);
        }

        return $this->reportService->exportFinancialExcel($companyId, $dateFrom, $dateTo);
    }

    public function exportCustomers(Request $request)
    {
        $format = $request->format ?? 'excel';
        $companyId = Auth::user()->company_id;

        if ($format === 'pdf') {
            return $this->reportService->exportCustomersPDF($companyId);
        }

        return $this->reportService->exportCustomersExcel($companyId);
    }

    public function exportCars(Request $request)
    {
        $format = $request->format ?? 'excel';
        $companyId = Auth::user()->company_id;

        if ($format === 'pdf') {
            return $this->reportService->exportCarsPDF($companyId);
        }

        return $this->reportService->exportCarsExcel($companyId);
    }

    private function getFinancialChartData($companyId, $dateFrom, $dateTo)
    {
        $days = [];
        $income = [];
        $expenses = [];

        $period = Carbon::parse($dateFrom);
        while ($period->lte($dateTo)) {
            $days[] = $period->format('d/m');
            
            $dayIncome = AccountReceivable::where('company_id', $companyId)
                ->whereDate('received_date', $period)
                ->sum('amount');
            
            $dayExpenses = AccountPayable::where('company_id', $companyId)
                ->whereDate('paid_date', $period)
                ->sum('amount');
            
            $income[] = $dayIncome;
            $expenses[] = $dayExpenses;
            
            $period->addDay();
        }

        return compact('days', 'income', 'expenses');
    }

    private function generateCashFlowData($companyId, $dateFrom, $dateTo)
    {
        $entries = collect();
        
        // Contas Recebidas
        $received = AccountReceivable::where('company_id', $companyId)
            ->where('status', 'received')
            ->whereBetween('received_date', [$dateFrom, $dateTo])
            ->get()
            ->map(function($item) {
                return [
                    'date' => $item->received_date,
                    'description' => $item->description,
                    'type' => 'income',
                    'amount' => $item->amount,
                    'category' => $item->category->name ?? 'Sem categoria',
                ];
            });

        // Contas Pagas
        $paid = AccountPayable::where('company_id', $companyId)
            ->where('status', 'paid')
            ->whereBetween('paid_date', [$dateFrom, $dateTo])
            ->get()
            ->map(function($item) {
                return [
                    'date' => $item->paid_date,
                    'description' => $item->description,
                    'type' => 'expense',
                    'amount' => $item->amount,
                    'category' => $item->category->name ?? 'Sem categoria',
                ];
            });

        return $entries->merge($received)->merge($paid)->sortBy('date');
    }
}
