<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Http\Requests\ScheduleRequest;
use App\Services\ScheduleService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ScheduleController extends Controller
{
    protected $scheduleService;

    public function __construct(ScheduleService $scheduleService)
    {
        $this->scheduleService = $scheduleService;
    }

    public function index(Request $request)
    {
        $query = Schedule::forUser(Auth::id())
            ->with('user');

        // Filtros
        if ($request->filled('type')) {
            $query->byType($request->type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('priority')) {
            $query->byPriority($request->priority);
        }

        if ($request->filled('date_from')) {
            $query->where('scheduled_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('scheduled_at', '<=', $request->date_to . ' 23:59:59');
        }

        $schedules = $query->orderBy('scheduled_at')->paginate(20);
        $statistics = $this->scheduleService->getStatistics();
        $upcoming = $this->scheduleService->getUpcoming(null, 5);
        $overdue = $this->scheduleService->getOverdue();

        return view('schedules.index', compact('schedules', 'statistics', 'upcoming', 'overdue'));
    }

    public function calendar(Request $request)
    {
        if ($request->ajax()) {
            $start = Carbon::parse($request->start);
            $end = Carbon::parse($request->end);
            
            $events = $this->scheduleService->getCalendarEvents($start, $end);
            
            return response()->json($events);
        }

        $statistics = $this->scheduleService->getStatistics();
        
        return view('schedules.calendar', compact('statistics'));
    }

    public function create()
    {
        return view('schedules.create');
    }

    public function store(ScheduleRequest $request)
    {
        $schedule = $this->scheduleService->create($request->validated());

        // Criar eventos recorrentes se necessário
        if ($schedule->is_recurring) {
            $this->scheduleService->createRecurringSchedules($schedule);
        }

        return redirect()->route('schedules.index')
            ->with('success', 'Evento criado com sucesso!');
    }

    public function show(Schedule $schedule)
    {
        $this->authorize('view', $schedule);
        
        return view('schedules.show', compact('schedule'));
    }

    public function edit(Schedule $schedule)
    {
        $this->authorize('update', $schedule);
        
        return view('schedules.edit', compact('schedule'));
    }

    public function update(ScheduleRequest $request, Schedule $schedule)
    {
        $this->authorize('update', $schedule);

        $this->scheduleService->update($schedule, $request->validated());

        return redirect()->route('schedules.index')
            ->with('success', 'Evento atualizado com sucesso!');
    }

    public function destroy(Schedule $schedule)
    {
        $this->authorize('delete', $schedule);

        $schedule->delete();

        return redirect()->route('schedules.index')
            ->with('success', 'Evento excluído com sucesso!');
    }

    public function complete(Schedule $schedule)
    {
        $this->authorize('update', $schedule);

        $schedule->markAsCompleted();

        if (request()->ajax()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Evento marcado como concluído!');
    }

    public function cancel(Schedule $schedule)
    {
        $this->authorize('update', $schedule);

        $schedule->markAsCancelled();

        if (request()->ajax()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Evento cancelado!');
    }

    public function today()
    {
        $schedules = $this->scheduleService->getToday();
        $statistics = $this->scheduleService->getStatistics();

        return view('schedules.today', compact('schedules', 'statistics'));
    }

    public function upcoming()
    {
        $schedules = $this->scheduleService->getUpcoming(null, 50);
        $statistics = $this->scheduleService->getStatistics();

        return view('schedules.upcoming', compact('schedules', 'statistics'));
    }
}
