<?php

namespace App\Http\Controllers;

use App\Services\SettingsService;
use App\Services\NotificationService;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    protected $settingsService;
    protected $notificationService;

    public function __construct(SettingsService $settingsService, NotificationService $notificationService)
    {
        $this->settingsService = $settingsService;
        $this->notificationService = $notificationService;
    }

    public function index()
    {
        $companyId = Auth::user()->company_id;
        
        // Inicializar configurações padrão se não existirem
        $this->settingsService->initializeDefaults($companyId);
        
        $settings = $this->settingsService->getAll($companyId);
        
        return view('settings.index', compact('settings'));
    }

    public function update(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        $request->validate([
            'settings' => 'required|array',
        ]);

        foreach ($request->settings as $key => $value) {
            // Buscar configuração existente para obter o tipo
            $setting = Setting::where('key', $key)
                ->where('company_id', $companyId)
                ->first();

            if ($setting) {
                $type = $setting->type;
                
                // Tratar valores booleanos
                if ($type === 'boolean') {
                    $value = $request->has("settings.{$key}") ? true : false;
                }
                
                $this->settingsService->set($key, $value, $type, $companyId);
            }
        }

        // Upload de logo se fornecido
        if ($request->hasFile('company_logo')) {
            $file = $request->file('company_logo');
            $path = $file->store('logos', 'public');
            $this->settingsService->set('company_logo', $path, 'string', $companyId);
        }

        return back()->with('success', 'Configurações salvas com sucesso!');
    }

    public function testNotifications(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        // Criar notificação de teste
        $this->notificationService->createForUser(Auth::id(), [
            'title' => 'Teste de Notificação',
            'message' => 'Esta é uma notificação de teste do sistema.',
            'type' => 'info',
            'priority' => 'medium',
        ]);

        return back()->with('success', 'Notificação de teste enviada!');
    }

    public function runChecks(Request $request)
    {
        $companyId = Auth::user()->company_id;
        
        // Executar verificações de vencimento
        $this->notificationService->checkOverdueAccounts($companyId);
        $this->notificationService->checkDueSoonAccounts($companyId);

        return back()->with('success', 'Verificações executadas com sucesso!');
    }

    public function backup(Request $request)
    {
        // Implementar backup manual
        try {
            $backupService = app(\App\Services\BackupService::class);
            $filename = $backupService->createBackup(Auth::user()->company_id);
            
            return response()->download(storage_path('app/backups/' . $filename))
                ->deleteFileAfterSend();
                
        } catch (\Exception $e) {
            return back()->with('error', 'Erro ao criar backup: ' . $e->getMessage());
        }
    }

    public function clearCache(Request $request)
    {
        $this->settingsService->clearCache(Auth::user()->company_id);
        
        return back()->with('success', 'Cache limpo com sucesso!');
    }
}
