<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ScheduleRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string|max:2000',
            'type' => 'required|in:appointment,reminder,task,meeting,deadline',
            'scheduled_at' => 'required|date|after:now',
            'end_at' => 'nullable|date|after:scheduled_at',
            'priority' => 'required|in:low,medium,high,urgent',
            'all_day' => 'boolean',
            'location' => 'nullable|string|max:255',
            'participants' => 'nullable|array',
            'participants.*' => 'string|max:255',
            'reminders' => 'nullable|array',
            'reminders.*' => 'integer|min:1|max:10080', // máximo 1 semana
            'color' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            'is_recurring' => 'boolean',
            'recurrence_pattern' => 'nullable|required_if:is_recurring,true|in:daily,weekly,monthly,yearly',
            'recurrence_end' => 'nullable|required_if:is_recurring,true|date|after:scheduled_at',
        ];
    }

    public function messages(): array
    {
        return [
            'title.required' => 'O título é obrigatório.',
            'type.required' => 'O tipo do evento é obrigatório.',
            'scheduled_at.required' => 'A data e hora são obrigatórias.',
            'scheduled_at.after' => 'A data deve ser futura.',
            'end_at.after' => 'A data de término deve ser posterior ao início.',
            'priority.required' => 'A prioridade é obrigatória.',
            'color.regex' => 'A cor deve estar no formato hexadecimal.',
            'recurrence_pattern.required_if' => 'O padrão de recorrência é obrigatório para eventos recorrentes.',
            'recurrence_end.required_if' => 'A data final é obrigatória para eventos recorrentes.',
        ];
    }

    protected function prepareForValidation()
    {
        // Converter strings vazias em null
        if ($this->end_at === '') {
            $this->merge(['end_at' => null]);
        }
        
        if ($this->location === '') {
            $this->merge(['location' => null]);
        }

        // Processar participantes
        if ($this->participants_string) {
            $participants = array_filter(array_map('trim', explode(',', $this->participants_string)));
            $this->merge(['participants' => $participants]);
        }

        // Processar lembretes
        if ($this->reminders_string) {
            $reminders = array_filter(array_map('intval', explode(',', $this->reminders_string)));
            $this->merge(['reminders' => $reminders]);
        }
    }
}
