<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Car extends Model
{
    use HasFactory;

    protected $fillable = [
        'model',
        'brand',
        'year',
        'plate',
        'color',
        'purchase_price',
        'purchase_date',
        'sale_price',
        'sale_date',
        'status',
        'notes',
        'photos',
        'company_id',
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'purchase_date' => 'date',
        'sale_date' => 'date',
        'photos' => 'array',
    ];

    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function expenses(): HasMany
    {
        return $this->hasMany(CarExpense::class);
    }

    public function getTotalExpensesAttribute(): float
    {
        return $this->expenses()->sum('amount');
    }

    public function getProfitLossAttribute(): float
    {
        if (!$this->sale_price || !$this->purchase_price) {
            return 0;
        }

        return $this->sale_price - $this->purchase_price - $this->getTotalExpensesAttribute();
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'owned' => '<span class="badge bg-success">Próprio</span>',
            'sold' => '<span class="badge bg-info">Vendido</span>',
            'maintenance' => '<span class="badge bg-warning">Manutenção</span>',
            default => '<span class="badge bg-secondary">Indefinido</span>',
        };
    }
}
