<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class DefaultCategory extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'type',
        'group',
        'icon',
        'color',
        'keywords',
        'usage_count',
        'is_popular',
        'is_active',
    ];

    protected $casts = [
        'keywords' => 'array',
        'is_popular' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function categories(): HasMany
    {
        return $this->hasMany(Category::class);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopePopular($query)
    {
        return $query->where('is_popular', true);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeByGroup($query, $group)
    {
        return $query->where('group', $group);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%")
              ->orWhereJsonContains('keywords', $search);
        });
    }

    public function incrementUsage()
    {
        $this->increment('usage_count');
        
        // Marcar como popular se usado mais de 10 vezes
        if ($this->usage_count >= 10 && !$this->is_popular) {
            $this->update(['is_popular' => true]);
        }
    }

    public function getKeywordsStringAttribute(): string
    {
        return is_array($this->keywords) ? implode(', ', $this->keywords) : '';
    }
}
