<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'message',
        'type',
        'priority',
        'data',
        'scheduled_for',
        'read_at',
        'is_system',
        'action_url',
        'action_text',
        'user_id',
        'company_id',
    ];

    protected $casts = [
        'data' => 'array',
        'scheduled_for' => 'datetime',
        'read_at' => 'datetime',
        'is_system' => 'boolean',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    public function scopeRead($query)
    {
        return $query->whereNotNull('read_at');
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeScheduled($query)
    {
        return $query->whereNotNull('scheduled_for')
                    ->where('scheduled_for', '<=', now());
    }

    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    public function getIsReadAttribute(): bool
    {
        return !is_null($this->read_at);
    }

    public function getTypeIconAttribute(): string
    {
        return match($this->type) {
            'info' => 'bi-info-circle',
            'warning' => 'bi-exclamation-triangle',
            'error' => 'bi-x-circle',
            'success' => 'bi-check-circle',
            default => 'bi-bell',
        };
    }

    public function getTypeBadgeAttribute(): string
    {
        return match($this->type) {
            'info' => 'bg-info',
            'warning' => 'bg-warning',
            'error' => 'bg-danger',
            'success' => 'bg-success',
            default => 'bg-secondary',
        };
    }
}
