<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Schedule extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'type',
        'scheduled_at',
        'end_at',
        'status',
        'priority',
        'all_day',
        'location',
        'participants',
        'reminders',
        'color',
        'metadata',
        'is_recurring',
        'recurrence_pattern',
        'recurrence_end',
        'user_id',
        'company_id',
    ];

    protected $casts = [
        'scheduled_at' => 'datetime',
        'end_at' => 'datetime',
        'recurrence_end' => 'date',
        'participants' => 'array',
        'reminders' => 'array',
        'metadata' => 'array',
        'all_day' => 'boolean',
        'is_recurring' => 'boolean',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeUpcoming($query)
    {
        return $query->where('scheduled_at', '>=', now())
                    ->where('status', 'pending');
    }

    public function scopeOverdue($query)
    {
        return $query->where('scheduled_at', '<', now())
                    ->where('status', 'pending');
    }

    public function scopeToday($query)
    {
        return $query->whereDate('scheduled_at', today());
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('scheduled_at', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeByPriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    public function getTypeIconAttribute(): string
    {
        return match($this->type) {
            'appointment' => 'bi-calendar-check',
            'reminder' => 'bi-bell',
            'task' => 'bi-check-square',
            'meeting' => 'bi-people',
            'deadline' => 'bi-exclamation-triangle',
            default => 'bi-calendar-event',
        };
    }

    public function getTypeLabelAttribute(): string
    {
        return match($this->type) {
            'appointment' => 'Compromisso',
            'reminder' => 'Lembrete',
            'task' => 'Tarefa',
            'meeting' => 'Reunião',
            'deadline' => 'Prazo',
            default => 'Evento',
        };
    }

    public function getPriorityBadgeAttribute(): string
    {
        return match($this->priority) {
            'low' => 'bg-secondary',
            'medium' => 'bg-primary',
            'high' => 'bg-warning',
            'urgent' => 'bg-danger',
            default => 'bg-secondary',
        };
    }

    public function getPriorityLabelAttribute(): string
    {
        return match($this->priority) {
            'low' => 'Baixa',
            'medium' => 'Média',
            'high' => 'Alta',
            'urgent' => 'Urgente',
            default => 'Média',
        };
    }

    public function getIsOverdueAttribute(): bool
    {
        return $this->scheduled_at < now() && $this->status === 'pending';
    }

    public function getFormattedDateAttribute(): string
    {
        if ($this->all_day) {
            return $this->scheduled_at->format('d/m/Y');
        }
        
        return $this->scheduled_at->format('d/m/Y H:i');
    }

    public function getFormattedDurationAttribute(): string
    {
        if (!$this->end_at || $this->all_day) {
            return '';
        }
        
        $duration = $this->scheduled_at->diffInMinutes($this->end_at);
        
        if ($duration < 60) {
            return $duration . ' min';
        }
        
        $hours = floor($duration / 60);
        $minutes = $duration % 60;
        
        return $hours . 'h' . ($minutes > 0 ? ' ' . $minutes . 'min' : '');
    }

    public function markAsCompleted()
    {
        $this->update(['status' => 'completed']);
    }

    public function markAsCancelled()
    {
        $this->update(['status' => 'cancelled']);
    }

    public function shouldSendReminder($minutesBefore): bool
    {
        if (!$this->reminders || $this->status !== 'pending') {
            return false;
        }

        $reminderTime = $this->scheduled_at->subMinutes($minutesBefore);
        
        return in_array($minutesBefore, $this->reminders) && 
               now()->between($reminderTime, $reminderTime->addMinutes(5));
    }
}
