<?php

namespace App\Services;

use App\Models\BankReconciliation;
use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Smalot\PdfParser\Parser as PdfParser;
use League\Csv\Reader;
use Carbon\Carbon;

class BankImportService
{
    public function importFile($filePath, $fileType, $bankName, $bankAccountId = null, $creditCardId = null, $companyId)
    {
        $fullPath = Storage::path($filePath);
        
        switch ($fileType) {
            case 'csv':
                return $this->importCSV($fullPath, $bankName, $bankAccountId, $creditCardId, $companyId);
            case 'excel':
                return $this->importExcel($fullPath, $bankName, $bankAccountId, $creditCardId, $companyId);
            case 'pdf':
                return $this->importPDF($fullPath, $bankName, $bankAccountId, $creditCardId, $companyId);
            case 'txt':
                return $this->importTXT($fullPath, $bankName, $bankAccountId, $creditCardId, $companyId);
            default:
                throw new \Exception('Tipo de arquivo não suportado');
        }
    }

    private function importCSV($filePath, $bankName, $bankAccountId, $creditCardId, $companyId)
    {
        $csv = Reader::createFromPath($filePath, 'r');
        $csv->setHeaderOffset(0);
        
        $imported = 0;
        $duplicates = 0;
        
        foreach ($csv as $record) {
            $transaction = $this->parseCSVRecord($record, $bankName);
            
            if ($this->isDuplicate($transaction, $bankAccountId, $companyId)) {
                $duplicates++;
                continue;
            }
            
            BankReconciliation::create([
                'transaction_date' => $transaction['date'],
                'description' => $transaction['description'],
                'document_number' => $transaction['document'] ?? null,
                'amount' => abs($transaction['amount']),
                'type' => $transaction['amount'] >= 0 ? 'credit' : 'debit',
                'bank_reference' => $transaction['reference'] ?? null,
                'bank_account_id' => $bankAccountId,
                'import_file' => basename($filePath),
                'company_id' => $companyId,
            ]);
            
            $imported++;
        }
        
        return compact('imported', 'duplicates');
    }

    private function importExcel($filePath, $bankName, $bankAccountId, $creditCardId, $companyId)
    {
        $spreadsheet = IOFactory::load($filePath);
        $worksheet = $spreadsheet->getActiveSheet();
        $rows = $worksheet->toArray();
        
        $imported = 0;
        $duplicates = 0;
        
        // Pular cabeçalho
        array_shift($rows);
        
        foreach ($rows as $row) {
            if (empty($row[0])) continue;
            
            $transaction = $this->parseExcelRow($row, $bankName);
            
            if ($this->isDuplicate($transaction, $bankAccountId, $companyId)) {
                $duplicates++;
                continue;
            }
            
            BankReconciliation::create([
                'transaction_date' => $transaction['date'],
                'description' => $transaction['description'],
                'document_number' => $transaction['document'] ?? null,
                'amount' => abs($transaction['amount']),
                'type' => $transaction['amount'] >= 0 ? 'credit' : 'debit',
                'bank_reference' => $transaction['reference'] ?? null,
                'bank_account_id' => $bankAccountId,
                'import_file' => basename($filePath),
                'company_id' => $companyId,
            ]);
            
            $imported++;
        }
        
        return compact('imported', 'duplicates');
    }

    private function importPDF($filePath, $bankName, $bankAccountId, $creditCardId, $companyId)
    {
        $parser = new PdfParser();
        $pdf = $parser->parseFile($filePath);
        $text = $pdf->getText();
        
        $transactions = $this->parsePDFText($text, $bankName);
        
        $imported = 0;
        $duplicates = 0;
        
        foreach ($transactions as $transaction) {
            if ($this->isDuplicate($transaction, $bankAccountId, $companyId)) {
                $duplicates++;
                continue;
            }
            
            BankReconciliation::create([
                'transaction_date' => $transaction['date'],
                'description' => $transaction['description'],
                'document_number' => $transaction['document'] ?? null,
                'amount' => abs($transaction['amount']),
                'type' => $transaction['amount'] >= 0 ? 'credit' : 'debit',
                'bank_reference' => $transaction['reference'] ?? null,
                'bank_account_id' => $bankAccountId,
                'import_file' => basename($filePath),
                'company_id' => $companyId,
            ]);
            
            $imported++;
        }
        
        return compact('imported', 'duplicates');
    }

    private function parseCSVRecord($record, $bankName)
    {
        // Adaptar conforme formato do banco
        switch (strtolower($bankName)) {
            case 'itau':
                return $this->parseItauCSV($record);
            case 'bradesco':
                return $this->parseBradescoCSV($record);
            case 'santander':
                return $this->parseSantanderCSV($record);
            case 'bb':
            case 'banco do brasil':
                return $this->parseBBCSV($record);
            default:
                return $this->parseGenericCSV($record);
        }
    }

    private function parseItauCSV($record)
    {
        return [
            'date' => Carbon::createFromFormat('d/m/Y', $record['data']),
            'description' => $record['historico'] ?? $record['descricao'],
            'amount' => (float) str_replace(['.', ','], ['', '.'], $record['valor']),
            'document' => $record['documento'] ?? null,
            'reference' => $record['referencia'] ?? null,
        ];
    }

    private function parseBradescoCSV($record)
    {
        return [
            'date' => Carbon::createFromFormat('d/m/Y', $record['Data']),
            'description' => $record['Histórico'],
            'amount' => (float) str_replace(['.', ','], ['', '.'], $record['Valor']),
            'document' => $record['Documento'] ?? null,
        ];
    }

    private function parseGenericCSV($record)
    {
        // Formato genérico: Data, Descrição, Valor
        $keys = array_keys($record);
        return [
            'date' => Carbon::parse($record[$keys[0]]),
            'description' => $record[$keys[1]],
            'amount' => (float) str_replace(['.', ','], ['', '.'], $record[$keys[2]]),
        ];
    }

    private function parsePDFText($text, $bankName)
    {
        $transactions = [];
        
        // Padrões regex para diferentes bancos
        switch (strtolower($bankName)) {
            case 'itau':
                $pattern = '/(\d{2}\/\d{2}\/\d{4})\s+(.+?)\s+([\d\.,]+)([CD]?)/m';
                break;
            case 'bradesco':
                $pattern = '/(\d{2}\/\d{2})\s+(.+?)\s+([\d\.,]+)([+-]?)/m';
                break;
            default:
                $pattern = '/(\d{2}\/\d{2}\/\d{4})\s+(.+?)\s+([\d\.,]+)/m';
        }
        
        preg_match_all($pattern, $text, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $amount = (float) str_replace(['.', ','], ['', '.'], $match[3]);
            
            // Determinar se é débito ou crédito
            if (isset($match[4])) {
                $isCredit = in_array($match[4], ['C', '+']);
            } else {
                $isCredit = $amount >= 0;
            }
            
            $transactions[] = [
                'date' => Carbon::createFromFormat('d/m/Y', $match[1]),
                'description' => trim($match[2]),
                'amount' => $isCredit ? abs($amount) : -abs($amount),
            ];
        }
        
        return $transactions;
    }

    private function isDuplicate($transaction, $bankAccountId, $companyId)
    {
        return BankReconciliation::where('company_id', $companyId)
            ->where('bank_account_id', $bankAccountId)
            ->where('transaction_date', $transaction['date'])
            ->where('amount', abs($transaction['amount']))
            ->where('description', $transaction['description'])
            ->exists();
    }

    public function autoReconcile($companyId, $bankAccountId = null)
    {
        $query = BankReconciliation::where('company_id', $companyId)
            ->where('status', 'pending');
            
        if ($bankAccountId) {
            $query->where('bank_account_id', $bankAccountId);
        }
        
        $pendingTransactions = $query->get();
        $reconciled = 0;
        
        foreach ($pendingTransactions as $transaction) {
            // Tentar encontrar conta a pagar correspondente
            if ($transaction->type === 'debit') {
                $accountPayable = AccountPayable::where('company_id', $companyId)
                    ->where('status', 'pending')
                    ->where('amount', $transaction->amount)
                    ->whereBetween('due_date', [
                        $transaction->transaction_date->subDays(5),
                        $transaction->transaction_date->addDays(5)
                    ])
                    ->first();
                
                if ($accountPayable) {
                    $accountPayable->update([
                        'status' => 'paid',
                        'paid_date' => $transaction->transaction_date,
                    ]);
                    
                    $transaction->update([
                        'status' => 'reconciled',
                        'account_payable_id' => $accountPayable->id,
                        'notes' => 'Conciliação automática',
                    ]);
                    
                    $reconciled++;
                    continue;
                }
            }
            
            // Tentar encontrar conta a receber correspondente
            if ($transaction->type === 'credit') {
                $accountReceivable = AccountReceivable::where('company_id', $companyId)
                    ->where('status', 'pending')
                    ->where('amount', $transaction->amount)
                    ->whereBetween('due_date', [
                        $transaction->transaction_date->subDays(5),
                        $transaction->transaction_date->addDays(5)
                    ])
                    ->first();
                
                if ($accountReceivable) {
                    $accountReceivable->update([
                        'status' => 'received',
                        'received_date' => $transaction->transaction_date,
                    ]);
                    
                    $transaction->update([
                        'status' => 'reconciled',
                        'account_receivable_id' => $accountReceivable->id,
                        'notes' => 'Conciliação automática',
                    ]);
                    
                    $reconciled++;
                }
            }
        }
        
        return $reconciled;
    }
}
