<?php

namespace App\Services;

use App\Models\DefaultCategory;
use App\Models\Category;
use Illuminate\Support\Facades\Auth;

class CategoryLibraryService
{
    public function getDefaultCategories($type = null, $group = null, $popular = false)
    {
        $query = DefaultCategory::active()->orderBy('sort_order')->orderBy('name');

        if ($type) {
            $query->byType($type);
        }

        if ($group) {
            $query->byGroup($group);
        }

        if ($popular) {
            $query->popular();
        }

        return $query->get();
    }

    public function getGroupedCategories($type = null)
    {
        $query = DefaultCategory::active()->orderBy('sort_order')->orderBy('name');

        if ($type) {
            $query->byType($type);
        }

        return $query->get()->groupBy('group');
    }

    public function getPopularCategories($type = null, $limit = 10)
    {
        $query = DefaultCategory::active()->popular()->orderBy('sort_order');

        if ($type) {
            $query->byType($type);
        }

        return $query->limit($limit)->get();
    }

    public function searchCategories($search, $type = null)
    {
        $query = DefaultCategory::active();

        if ($type) {
            $query->byType($type);
        }

        return $query->where(function($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%")
              ->orWhereJsonContains('keywords', $search);
        })->orderBy('is_popular', 'desc')->orderBy('name')->get();
    }

    public function importToCompany($categoryIds, $companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;
        $imported = 0;
        $skipped = 0;

        foreach ($categoryIds as $categoryId) {
            $defaultCategory = DefaultCategory::find($categoryId);
            if (!$defaultCategory) {
                continue;
            }

            // Verificar se já existe
            $exists = Category::where('company_id', $companyId)
                ->where('name', $defaultCategory->name)
                ->where('type', $defaultCategory->type)
                ->exists();

            if ($exists) {
                $skipped++;
                continue;
            }

            // Criar categoria
            Category::create([
                'name' => $defaultCategory->name,
                'type' => $defaultCategory->type,
                'color' => $defaultCategory->color,
                'description' => $defaultCategory->description,
                'icon' => $defaultCategory->icon,
                'group' => $defaultCategory->group,
                'keywords' => $defaultCategory->keywords,
                'default_category_id' => $defaultCategory->id,
                'sort_order' => $defaultCategory->sort_order,
                'company_id' => $companyId,
            ]);

            $imported++;
        }

        return compact('imported', 'skipped');
    }

    public function importAllByGroup($group, $companyId = null)
    {
        $categories = DefaultCategory::active()->byGroup($group)->get();
        $categoryIds = $categories->pluck('id')->toArray();
        
        return $this->importToCompany($categoryIds, $companyId);
    }

    public function importPopular($type = null, $companyId = null)
    {
        $query = DefaultCategory::active()->popular();
        
        if ($type) {
            $query->byType($type);
        }
        
        $categories = $query->get();
        $categoryIds = $categories->pluck('id')->toArray();
        
        return $this->importToCompany($categoryIds, $companyId);
    }

    public function getCompanyCategories($companyId = null, $type = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;
        
        $query = Category::where('company_id', $companyId)
            ->with('defaultCategory')
            ->orderBy('sort_order')
            ->orderBy('name');

        if ($type) {
            $query->byType($type);
        }

        return $query->get();
    }

    public function getUsageStats($companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;

        return [
            'total_categories' => Category::where('company_id', $companyId)->count(),
            'debit_categories' => Category::where('company_id', $companyId)->where('type', 'debit')->count(),
            'credit_categories' => Category::where('company_id', $companyId)->where('type', 'credit')->count(),
            'most_used' => $this->getMostUsedCategories($companyId, 5),
            'least_used' => $this->getLeastUsedCategories($companyId, 5),
            'unused' => $this->getUnusedCategories($companyId),
        ];
    }

    private function getMostUsedCategories($companyId, $limit = 5)
    {
        return Category::where('company_id', $companyId)
            ->withCount([
                'customerEntries',
                'accountsPayable',
                'accountsReceivable',
                'carExpenses'
            ])
            ->get()
            ->map(function($category) {
                $category->total_usage = $category->customer_entries_count +
                                       $category->accounts_payable_count +
                                       $category->accounts_receivable_count +
                                       $category->car_expenses_count;
                return $category;
            })
            ->sortByDesc('total_usage')
            ->take($limit);
    }

    private function getLeastUsedCategories($companyId, $limit = 5)
    {
        return Category::where('company_id', $companyId)
            ->withCount([
                'customerEntries',
                'accountsPayable',
                'accountsReceivable',
                'carExpenses'
            ])
            ->get()
            ->map(function($category) {
                $category->total_usage = $category->customer_entries_count +
                                       $category->accounts_payable_count +
                                       $category->accounts_receivable_count +
                                       $category->car_expenses_count;
                return $category;
            })
            ->where('total_usage', '>', 0)
            ->sortBy('total_usage')
            ->take($limit);
    }

    private function getUnusedCategories($companyId)
    {
        return Category::where('company_id', $companyId)
            ->whereDoesntHave('customerEntries')
            ->whereDoesntHave('accountsPayable')
            ->whereDoesntHave('accountsReceivable')
            ->whereDoesntHave('carExpenses')
            ->count();
    }

    public function initializeDefaultCategories()
    {
        $categories = $this->getDefaultCategoriesData();

        foreach ($categories as $category) {
            DefaultCategory::firstOrCreate(
                [
                    'name' => $category['name'],
                    'type' => $category['type'],
                    'group' => $category['group'],
                ],
                $category
            );
        }
    }

    private function getDefaultCategoriesData(): array
    {
        return [
            // CATEGORIAS DE DÉBITO (DESPESAS)
            
            // Automotivo
            [
                'name' => 'Combustível',
                'type' => 'debit',
                'color' => '#ef4444',
                'description' => 'Gastos com combustível para veículos',
                'icon' => 'bi-fuel-pump',
                'group' => 'automotive',
                'keywords' => ['gasolina', 'etanol', 'diesel', 'álcool', 'posto'],
                'is_popular' => true,
                'sort_order' => 1,
            ],
            [
                'name' => 'Manutenção de Veículos',
                'type' => 'debit',
                'color' => '#f97316',
                'description' => 'Reparos e manutenção de carros',
                'icon' => 'bi-wrench',
                'group' => 'automotive',
                'keywords' => ['mecânico', 'reparo', 'conserto', 'oficina', 'peças'],
                'is_popular' => true,
                'sort_order' => 2,
            ],
            [
                'name' => 'Seguro Veicular',
                'type' => 'debit',
                'color' => '#3b82f6',
                'description' => 'Seguros de veículos',
                'icon' => 'bi-shield-check',
                'group' => 'automotive',
                'keywords' => ['seguro', 'proteção', 'cobertura'],
                'is_popular' => true,
                'sort_order' => 3,
            ],
            [
                'name' => 'IPVA',
                'type' => 'debit',
                'color' => '#8b5cf6',
                'description' => 'Imposto sobre Propriedade de Veículos Automotores',
                'icon' => 'bi-file-text',
                'group' => 'automotive',
                'keywords' => ['imposto', 'ipva', 'veicular'],
                'is_popular' => true,
                'sort_order' => 4,
            ],
            [
                'name' => 'Licenciamento',
                'type' => 'debit',
                'color' => '#06b6d4',
                'description' => 'Taxa de licenciamento anual',
                'icon' => 'bi-card-text',
                'group' => 'automotive',
                'keywords' => ['licenciamento', 'taxa', 'detran'],
                'is_popular' => false,
                'sort_order' => 5,
            ],
            [
                'name' => 'Multas de Trânsito',
                'type' => 'debit',
                'color' => '#dc2626',
                'description' => 'Multas e infrações de trânsito',
                'icon' => 'bi-exclamation-triangle',
                'group' => 'automotive',
                'keywords' => ['multa', 'infração', 'trânsito'],
                'is_popular' => false,
                'sort_order' => 6,
            ],
            [
                'name' => 'Lavagem e Estética',
                'type' => 'debit',
                'color' => '#10b981',
                'description' => 'Lavagem e cuidados estéticos dos veículos',
                'icon' => 'bi-droplet',
                'group' => 'automotive',
                'keywords' => ['lavagem', 'enceramento', 'limpeza'],
                'is_popular' => false,
                'sort_order' => 7,
            ],

            // Escritório
            [
                'name' => 'Material de Escritório',
                'type' => 'debit',
                'color' => '#6366f1',
                'description' => 'Papelaria e materiais de escritório',
                'icon' => 'bi-pencil',
                'group' => 'office',
                'keywords' => ['papel', 'caneta', 'grampeador', 'papelaria'],
                'is_popular' => true,
                'sort_order' => 10,
            ],
            [
                'name' => 'Telefone e Internet',
                'type' => 'debit',
                'color' => '#059669',
                'description' => 'Contas de telefone e internet',
                'icon' => 'bi-telephone',
                'group' => 'office',
                'keywords' => ['telefone', 'internet', 'comunicação'],
                'is_popular' => true,
                'sort_order' => 11,
            ],
            [
                'name' => 'Energia Elétrica',
                'type' => 'debit',
                'color' => '#eab308',
                'description' => 'Conta de luz',
                'icon' => 'bi-lightning',
                'group' => 'office',
                'keywords' => ['luz', 'energia', 'eletricidade'],
                'is_popular' => true,
                'sort_order' => 12,
            ],
            [
                'name' => 'Água',
                'type' => 'debit',
                'color' => '#0ea5e9',
                'description' => 'Conta de água',
                'icon' => 'bi-droplet-fill',
                'group' => 'office',
                'keywords' => ['água', 'saneamento'],
                'is_popular' => true,
                'sort_order' => 13,
            ],
            [
                'name' => 'Aluguel',
                'type' => 'debit',
                'color' => '#7c3aed',
                'description' => 'Aluguel de imóveis',
                'icon' => 'bi-house',
                'group' => 'office',
                'keywords' => ['aluguel', 'locação', 'imóvel'],
                'is_popular' => true,
                'sort_order' => 14,
            ],

            // Tecnologia
            [
                'name' => 'Software e Licenças',
                'type' => 'debit',
                'color' => '#6366f1',
                'description' => 'Licenças de software e aplicativos',
                'icon' => 'bi-laptop',
                'group' => 'technology',
                'keywords' => ['software', 'licença', 'aplicativo'],
                'is_popular' => true,
                'sort_order' => 20,
            ],
            [
                'name' => 'Equipamentos de TI',
                'type' => 'debit',
                'color' => '#374151',
                'description' => 'Computadores, impressoras e equipamentos',
                'icon' => 'bi-pc-display',
                'group' => 'technology',
                'keywords' => ['computador', 'impressora', 'equipamento'],
                'is_popular' => false,
                'sort_order' => 21,
            ],
            [
                'name' => 'Hospedagem e Domínio',
                'type' => 'debit',
                'color' => '#059669',
                'description' => 'Hospedagem de sites e domínios',
                'icon' => 'bi-globe',
                'group' => 'technology',
                'keywords' => ['hospedagem', 'domínio', 'site'],
                'is_popular' => false,
                'sort_order' => 22,
            ],

            // Marketing
            [
                'name' => 'Publicidade e Propaganda',
                'type' => 'debit',
                'color' => '#ec4899',
                'description' => 'Gastos com marketing e publicidade',
                'icon' => 'bi-megaphone',
                'group' => 'marketing',
                'keywords' => ['publicidade', 'propaganda', 'marketing'],
                'is_popular' => true,
                'sort_order' => 30,
            ],
            [
                'name' => 'Redes Sociais',
                'type' => 'debit',
                'color' => '#3b82f6',
                'description' => 'Impulsionamento e ads em redes sociais',
                'icon' => 'bi-share',
                'group' => 'marketing',
                'keywords' => ['facebook', 'instagram', 'google ads'],
                'is_popular' => true,
                'sort_order' => 31,
            ],

            // Financeiro
            [
                'name' => 'Taxas Bancárias',
                'type' => 'debit',
                'color' => '#dc2626',
                'description' => 'Taxas e tarifas bancárias',
                'icon' => 'bi-bank',
                'group' => 'finance',
                'keywords' => ['banco', 'taxa', 'tarifa'],
                'is_popular' => true,
                'sort_order' => 40,
            ],
            [
                'name' => 'Juros e Multas',
                'type' => 'debit',
                'color' => '#ef4444',
                'description' => 'Juros de financiamentos e multas',
                'icon' => 'bi-percent',
                'group' => 'finance',
                'keywords' => ['juros', 'multa', 'financiamento'],
                'is_popular' => false,
                'sort_order' => 41,
            ],
            [
                'name' => 'Contabilidade',
                'type' => 'debit',
                'color' => '#6366f1',
                'description' => 'Serviços contábeis',
                'icon' => 'bi-calculator',
                'group' => 'finance',
                'keywords' => ['contador', 'contabilidade', 'imposto'],
                'is_popular' => true,
                'sort_order' => 42,
            ],

            // Alimentação
            [
                'name' => 'Alimentação',
                'type' => 'debit',
                'color' => '#f59e0b',
                'description' => 'Gastos com alimentação',
                'icon' => 'bi-cup-straw',
                'group' => 'food',
                'keywords' => ['comida', 'restaurante', 'lanche'],
                'is_popular' => true,
                'sort_order' => 50,
            ],

            // Viagem
            [
                'name' => 'Viagens e Hospedagem',
                'type' => 'debit',
                'color' => '#8b5cf6',
                'description' => 'Gastos com viagens de negócios',
                'icon' => 'bi-airplane',
                'group' => 'travel',
                'keywords' => ['viagem', 'hotel', 'passagem'],
                'is_popular' => false,
                'sort_order' => 60,
            ],

            // CATEGORIAS DE CRÉDITO (RECEITAS)
            
            // Vendas
            [
                'name' => 'Venda de Veículos',
                'type' => 'credit',
                'color' => '#10b981',
                'description' => 'Receita com venda de carros',
                'icon' => 'bi-car-front',
                'group' => 'sales',
                'keywords' => ['venda', 'carro', 'veículo'],
                'is_popular' => true,
                'sort_order' => 100,
            ],
            [
                'name' => 'Venda de Produtos',
                'type' => 'credit',
                'color' => '#059669',
                'description' => 'Receita com venda de produtos',
                'icon' => 'bi-box',
                'group' => 'sales',
                'keywords' => ['produto', 'mercadoria', 'venda'],
                'is_popular' => true,
                'sort_order' => 101,
            ],

            // Serviços
            [
                'name' => 'Prestação de Serviços',
                'type' => 'credit',
                'color' => '#3b82f6',
                'description' => 'Receita com prestação de serviços',
                'icon' => 'bi-tools',
                'group' => 'services',
                'keywords' => ['serviço', 'mão de obra', 'trabalho'],
                'is_popular' => true,
                'sort_order' => 110,
            ],
            [
                'name' => 'Consultoria',
                'type' => 'credit',
                'color' => '#6366f1',
                'description' => 'Receita com consultoria',
                'icon' => 'bi-person-check',
                'group' => 'consulting',
                'keywords' => ['consultoria', 'assessoria', 'orientação'],
                'is_popular' => false,
                'sort_order' => 111,
            ],

            // Aluguel
            [
                'name' => 'Receita de Aluguel',
                'type' => 'credit',
                'color' => '#7c3aed',
                'description' => 'Receita com locação de imóveis',
                'icon' => 'bi-house-check',
                'group' => 'rental',
                'keywords' => ['aluguel', 'locação', 'renda'],
                'is_popular' => false,
                'sort_order' => 120,
            ],

            // Comissões
            [
                'name' => 'Comissões',
                'type' => 'credit',
                'color' => '#ec4899',
                'description' => 'Receita com comissões',
                'icon' => 'bi-percent',
                'group' => 'commission',
                'keywords' => ['comissão', 'percentual', 'intermediação'],
                'is_popular' => true,
                'sort_order' => 130,
            ],

            // Financeiro
            [
                'name' => 'Juros Recebidos',
                'type' => 'credit',
                'color' => '#10b981',
                'description' => 'Juros de aplicações financeiras',
                'icon' => 'bi-graph-up',
                'group' => 'finance',
                'keywords' => ['juros', 'rendimento', 'aplicação'],
                'is_popular' => false,
                'sort_order' => 140,
            ],
            [
                'name' => 'Outras Receitas',
                'type' => 'credit',
                'color' => '#6b7280',
                'description' => 'Outras receitas diversas',
                'icon' => 'bi-plus-circle',
                'group' => 'finance',
                'keywords' => ['outras', 'diversas', 'extras'],
                'is_popular' => false,
                'sort_order' => 141,
            ],
        ];
    }

    public function getAvailableCategories($filters = [])
    {
        $query = DefaultCategory::active();

        if (isset($filters['type'])) {
            $query->byType($filters['type']);
        }

        if (isset($filters['group'])) {
            $query->byGroup($filters['group']);
        }

        if (isset($filters['search'])) {
            $query->search($filters['search']);
        }

        if (isset($filters['popular']) && $filters['popular']) {
            $query->popular();
        }

        return $query->orderBy('group')->orderBy('name')->get();
    }

    public function getGroups()
    {
        return DefaultCategory::active()
            ->select('group')
            ->distinct()
            ->whereNotNull('group')
            ->orderBy('group')
            ->pluck('group');
    }

    public function importCategory($defaultCategoryId, $companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;
        $defaultCategory = DefaultCategory::findOrFail($defaultCategoryId);

        // Verificar se já existe
        $existing = Category::where('company_id', $companyId)
            ->where('default_category_id', $defaultCategoryId)
            ->first();

        if ($existing) {
            return ['success' => false, 'message' => 'Categoria já importada'];
        }

        $category = Category::create([
            'name' => $defaultCategory->name,
            'description' => $defaultCategory->description,
            'type' => $defaultCategory->type,
            'color' => $defaultCategory->color,
            'icon' => $defaultCategory->icon,
            'keywords' => $defaultCategory->keywords,
            'default_category_id' => $defaultCategory->id,
            'company_id' => $companyId,
        ]);

        $defaultCategory->incrementUsage();

        return ['success' => true, 'category' => $category];
    }

    public function importMultiple($defaultCategoryIds, $companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;
        $imported = [];
        $skipped = [];

        foreach ($defaultCategoryIds as $id) {
            $result = $this->importCategory($id, $companyId);
            
            if ($result['success']) {
                $imported[] = $result['category'];
            } else {
                $skipped[] = $id;
            }
        }

        return [
            'imported' => count($imported),
            'skipped' => count($skipped),
            'categories' => $imported
        ];
    }

    public function importByGroup($group, $companyId = null)
    {
        $categories = DefaultCategory::active()
            ->byGroup($group)
            ->pluck('id')
            ->toArray();

        return $this->importMultiple($categories, $companyId);
    }

    public function getImportedCategories($companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;

        return Category::where('company_id', $companyId)
            ->whereNotNull('default_category_id')
            ->pluck('default_category_id')
            ->toArray();
    }

    public function getStatistics($companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;

        $totalDefault = DefaultCategory::active()->count();
        $imported = Category::where('company_id', $companyId)
            ->whereNotNull('default_category_id')
            ->count();
        $custom = Category::where('company_id', $companyId)
            ->whereNull('default_category_id')
            ->count();

        return [
            'total_available' => $totalDefault,
            'imported' => $imported,
            'custom' => $custom,
            'import_percentage' => $totalDefault > 0 ? round(($imported / $totalDefault) * 100, 1) : 0,
        ];
    }
}
