<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;

class NotificationService
{
    protected $settingsService;

    public function __construct(SettingsService $settingsService)
    {
        $this->settingsService = $settingsService;
    }

    public function create(array $data): Notification
    {
        return Notification::create([
            'title' => $data['title'],
            'message' => $data['message'],
            'type' => $data['type'] ?? 'info',
            'priority' => $data['priority'] ?? 'medium',
            'data' => $data['data'] ?? null,
            'scheduled_for' => $data['scheduled_for'] ?? null,
            'action_url' => $data['action_url'] ?? null,
            'action_text' => $data['action_text'] ?? null,
            'user_id' => $data['user_id'],
            'company_id' => $data['company_id'],
            'is_system' => $data['is_system'] ?? false,
        ]);
    }

    public function createForUser($userId, array $data): Notification
    {
        $user = User::find($userId);
        $data['user_id'] = $userId;
        $data['company_id'] = $user->company_id;
        
        return $this->create($data);
    }

    public function createForCompany($companyId, array $data): array
    {
        $users = User::where('company_id', $companyId)->get();
        $notifications = [];
        
        foreach ($users as $user) {
            $data['user_id'] = $user->id;
            $data['company_id'] = $companyId;
            $notifications[] = $this->create($data);
        }
        
        return $notifications;
    }

    public function getUnreadCount($userId = null): int
    {
        $userId = $userId ?? Auth::id();
        
        return Notification::where('user_id', $userId)
            ->whereNull('read_at')
            ->count();
    }

    public function getRecent($userId = null, $limit = 10)
    {
        $userId = $userId ?? Auth::id();
        
        return Notification::where('user_id', $userId)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    public function markAsRead($notificationId, $userId = null)
    {
        $userId = $userId ?? Auth::id();
        
        return Notification::where('id', $notificationId)
            ->where('user_id', $userId)
            ->update(['read_at' => now()]);
    }

    public function markAllAsRead($userId = null)
    {
        $userId = $userId ?? Auth::id();
        
        return Notification::where('user_id', $userId)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);
    }

    public function checkOverdueAccounts($companyId = null)
    {
        if (!$this->settingsService->get('notify_overdue', true, $companyId)) {
            return;
        }

        // Contas a Pagar vencidas
        $overduePayables = AccountPayable::where('company_id', $companyId)
            ->where('status', 'pending')
            ->where('due_date', '<', now()->startOfDay())
            ->get();

        foreach ($overduePayables as $account) {
            $this->createForCompany($companyId, [
                'title' => 'Conta a Pagar Vencida',
                'message' => "A conta '{$account->description}' venceu em {$account->due_date->format('d/m/Y')}",
                'type' => 'error',
                'priority' => 'high',
                'action_url' => route('accounts-payable.show', $account),
                'action_text' => 'Ver Conta',
                'data' => [
                    'account_type' => 'payable',
                    'account_id' => $account->id,
                    'amount' => $account->amount,
                ],
            ]);
        }

        // Contas a Receber vencidas
        $overdueReceivables = AccountReceivable::where('company_id', $companyId)
            ->where('status', 'pending')
            ->where('due_date', '<', now()->startOfDay())
            ->get();

        foreach ($overdueReceivables as $account) {
            $this->createForCompany($companyId, [
                'title' => 'Conta a Receber Vencida',
                'message' => "A conta '{$account->description}' venceu em {$account->due_date->format('d/m/Y')}",
                'type' => 'warning',
                'priority' => 'high',
                'action_url' => route('accounts-receivable.show', $account),
                'action_text' => 'Ver Conta',
                'data' => [
                    'account_type' => 'receivable',
                    'account_id' => $account->id,
                    'amount' => $account->amount,
                ],
            ]);
        }
    }

    public function checkDueSoonAccounts($companyId = null)
    {
        if (!$this->settingsService->get('notify_due_soon', true, $companyId)) {
            return;
        }

        $dueSoonDays = (int) $this->settingsService->get('due_soon_days', 7, $companyId);
        $dueDate = now()->addDays($dueSoonDays)->endOfDay();

        // Contas a Pagar com vencimento próximo
        $dueSoonPayables = AccountPayable::where('company_id', $companyId)
            ->where('status', 'pending')
            ->whereBetween('due_date', [now()->startOfDay(), $dueDate])
            ->get();

        foreach ($dueSoonPayables as $account) {
            $this->createForCompany($companyId, [
                'title' => 'Vencimento Próximo - Conta a Pagar',
                'message' => "A conta '{$account->description}' vence em {$account->due_date->format('d/m/Y')}",
                'type' => 'warning',
                'priority' => 'medium',
                'action_url' => route('accounts-payable.show', $account),
                'action_text' => 'Ver Conta',
                'data' => [
                    'account_type' => 'payable',
                    'account_id' => $account->id,
                    'amount' => $account->amount,
                ],
            ]);
        }

        // Contas a Receber com vencimento próximo
        $dueSoonReceivables = AccountReceivable::where('company_id', $companyId)
            ->where('status', 'pending')
            ->whereBetween('due_date', [now()->startOfDay(), $dueDate])
            ->get();

        foreach ($dueSoonReceivables as $account) {
            $this->createForCompany($companyId, [
                'title' => 'Vencimento Próximo - Conta a Receber',
                'message' => "A conta '{$account->description}' vence em {$account->due_date->format('d/m/Y')}",
                'type' => 'info',
                'priority' => 'medium',
                'action_url' => route('accounts-receivable.show', $account),
                'action_text' => 'Ver Conta',
                'data' => [
                    'account_type' => 'receivable',
                    'account_id' => $account->id,
                    'amount' => $account->amount,
                ],
            ]);
        }
    }

    public function sendEmailNotification(Notification $notification)
    {
        if (!$this->settingsService->get('email_notifications', false, $notification->company_id)) {
            return;
        }

        $email = $this->settingsService->get('notification_email', null, $notification->company_id);
        if (!$email) {
            return;
        }

        // Implementar envio de e-mail
        // Mail::to($email)->send(new NotificationMail($notification));
    }
}
