<?php

namespace App\Services;

use App\Models\AccountPayable;
use App\Models\AccountReceivable;
use App\Models\Customer;
use App\Models\Car;
use App\Models\Company;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class ReportService
{
    public function exportFinancialExcel($companyId, $dateFrom, $dateTo)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Configurar cabeçalho
        $company = Company::find($companyId);
        $sheet->setCellValue('A1', 'RELATÓRIO FINANCEIRO');
        $sheet->setCellValue('A2', $company->name);
        $sheet->setCellValue('A3', 'Período: ' . $dateFrom->format('d/m/Y') . ' a ' . $dateTo->format('d/m/Y'));
        
        // Estilo do cabeçalho
        $sheet->getStyle('A1:A3')->getFont()->setBold(true);
        $sheet->getStyle('A1')->getFont()->setSize(16);
        
        // Contas a Pagar
        $row = 6;
        $sheet->setCellValue('A' . $row, 'CONTAS A PAGAR');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true);
        $row++;
        
        $sheet->setCellValue('A' . $row, 'Data Venc.');
        $sheet->setCellValue('B' . $row, 'Descrição');
        $sheet->setCellValue('C' . $row, 'Fornecedor');
        $sheet->setCellValue('D' . $row, 'Valor');
        $sheet->setCellValue('E' . $row, 'Status');
        $sheet->getStyle('A' . $row . ':E' . $row)->getFont()->setBold(true);
        $row++;
        
        $accountsPayable = AccountPayable::where('company_id', $companyId)
            ->with(['supplier', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();
        
        foreach ($accountsPayable as $account) {
            $sheet->setCellValue('A' . $row, $account->due_date->format('d/m/Y'));
            $sheet->setCellValue('B' . $row, $account->description);
            $sheet->setCellValue('C' . $row, $account->supplier->name ?? '-');
            $sheet->setCellValue('D' . $row, $account->amount);
            $sheet->setCellValue('E' . $row, $this->getStatusText($account->status));
            $row++;
        }
        
        // Contas a Receber
        $row += 2;
        $sheet->setCellValue('A' . $row, 'CONTAS A RECEBER');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true);
        $row++;
        
        $sheet->setCellValue('A' . $row, 'Data Venc.');
        $sheet->setCellValue('B' . $row, 'Descrição');
        $sheet->setCellValue('C' . $row, 'Cliente');
        $sheet->setCellValue('D' . $row, 'Valor');
        $sheet->setCellValue('E' . $row, 'Status');
        $sheet->getStyle('A' . $row . ':E' . $row)->getFont()->setBold(true);
        $row++;
        
        $accountsReceivable = AccountReceivable::where('company_id', $companyId)
            ->with(['customer', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();
        
        foreach ($accountsReceivable as $account) {
            $sheet->setCellValue('A' . $row, $account->due_date->format('d/m/Y'));
            $sheet->setCellValue('B' . $row, $account->description);
            $sheet->setCellValue('C' . $row, $account->customer->name ?? '-');
            $sheet->setCellValue('D' . $row, $account->amount);
            $sheet->setCellValue('E' . $row, $this->getStatusText($account->status));
            $row++;
        }
        
        // Resumo
        $row += 2;
        $sheet->setCellValue('A' . $row, 'RESUMO');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true);
        $row++;
        
        $totalPayable = $accountsPayable->sum('amount');
        $totalReceivable = $accountsReceivable->sum('amount');
        $balance = $totalReceivable - $totalPayable;
        
        $sheet->setCellValue('A' . $row, 'Total a Pagar:');
        $sheet->setCellValue('B' . $row, $totalPayable);
        $row++;
        $sheet->setCellValue('A' . $row, 'Total a Receber:');
        $sheet->setCellValue('B' . $row, $totalReceivable);
        $row++;
        $sheet->setCellValue('A' . $row, 'Saldo:');
        $sheet->setCellValue('B' . $row, $balance);
        
        // Formatação de moeda
        $sheet->getStyle('D:D')->getNumberFormat()->setFormatCode('_("R$"* #,##0.00_);_("R$"* $$#,##0.00$$;_("R$"* "-"??_);_(@_)');
        $sheet->getStyle('B' . ($row-2) . ':B' . $row)->getNumberFormat()->setFormatCode('_("R$"* #,##0.00_);_("R$"* $$#,##0.00$$;_("R$"* "-"??_);_(@_)');
        
        // Auto ajustar colunas
        foreach (range('A', 'E') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new Xlsx($spreadsheet);
        $filename = 'relatorio_financeiro_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $writer->save('php://output');
        exit;
    }

    public function exportFinancialPDF($companyId, $dateFrom, $dateTo)
    {
        $company = Company::find($companyId);
        
        $accountsPayable = AccountPayable::where('company_id', $companyId)
            ->with(['supplier', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();
        
        $accountsReceivable = AccountReceivable::where('company_id', $companyId)
            ->with(['customer', 'category'])
            ->whereBetween('due_date', [$dateFrom, $dateTo])
            ->orderBy('due_date')
            ->get();
        
        $summary = [
            'total_payable' => $accountsPayable->sum('amount'),
            'total_receivable' => $accountsReceivable->sum('amount'),
        ];
        $summary['balance'] = $summary['total_receivable'] - $summary['total_payable'];
        
        $data = compact('company', 'accountsPayable', 'accountsReceivable', 'summary', 'dateFrom', 'dateTo');
        
        $pdf = Pdf::loadView('reports.pdf.financial', $data);
        $pdf->setPaper('A4', 'portrait');
        
        return $pdf->download('relatorio_financeiro_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    public function exportCustomersExcel($companyId)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        $company = Company::find($companyId);
        $sheet->setCellValue('A1', 'RELATÓRIO DE CLIENTES');
        $sheet->setCellValue('A2', $company->name);
        $sheet->setCellValue('A3', 'Data: ' . now()->format('d/m/Y H:i'));
        
        $sheet->getStyle('A1:A3')->getFont()->setBold(true);
        $sheet->getStyle('A1')->getFont()->setSize(16);
        
        $row = 6;
        $sheet->setCellValue('A' . $row, 'Nome');
        $sheet->setCellValue('B' . $row, 'E-mail');
        $sheet->setCellValue('C' . $row, 'Telefone');
        $sheet->setCellValue('D' . $row, 'CPF/CNPJ');
        $sheet->setCellValue('E' . $row, 'Saldo Atual');
        $sheet->setCellValue('F' . $row, 'Limite Crédito');
        $sheet->setCellValue('G' . $row, 'Status');
        $sheet->getStyle('A' . $row . ':G' . $row)->getFont()->setBold(true);
        $row++;
        
        $customers = Customer::where('company_id', $companyId)->get();
        
        foreach ($customers as $customer) {
            $sheet->setCellValue('A' . $row, $customer->name);
            $sheet->setCellValue('B' . $row, $customer->email ?? '-');
            $sheet->setCellValue('C' . $row, $customer->phone ?? '-');
            $sheet->setCellValue('D' . $row, $customer->cpf_cnpj ?? '-');
            $sheet->setCellValue('E' . $row, $customer->current_balance);
            $sheet->setCellValue('F' . $row, $customer->credit_limit);
            $sheet->setCellValue('G' . $row, $customer->active ? 'Ativo' : 'Inativo');
            $row++;
        }
        
        // Formatação
        $sheet->getStyle('E:F')->getNumberFormat()->setFormatCode('_("R$"* #,##0.00_);_("R$"* $$#,##0.00$$;_("R$"* "-"??_);_(@_)');
        
        foreach (range('A', 'G') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new Xlsx($spreadsheet);
        $filename = 'relatorio_clientes_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $writer->save('php://output');
        exit;
    }

    public function exportCarsExcel($companyId)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        $company = Company::find($companyId);
        $sheet->setCellValue('A1', 'RELATÓRIO DE VEÍCULOS');
        $sheet->setCellValue('A2', $company->name);
        $sheet->setCellValue('A3', 'Data: ' . now()->format('d/m/Y H:i'));
        
        $sheet->getStyle('A1:A3')->getFont()->setBold(true);
        $sheet->getStyle('A1')->getFont()->setSize(16);
        
        $row = 6;
        $sheet->setCellValue('A' . $row, 'Marca/Modelo');
        $sheet->setCellValue('B' . $row, 'Ano');
        $sheet->setCellValue('C' . $row, 'Placa');
        $sheet->setCellValue('D' . $row, 'Preço Compra');
        $sheet->setCellValue('E' . $row, 'Preço Venda');
        $sheet->setCellValue('F' . $row, 'Total Despesas');
        $sheet->setCellValue('G' . $row, 'Lucro/Prejuízo');
        $sheet->setCellValue('H' . $row, 'Status');
        $sheet->getStyle('A' . $row . ':H' . $row)->getFont()->setBold(true);
        $row++;
        
        $cars = Car::where('company_id', $companyId)
            ->withSum('expenses', 'amount')
            ->get();
        
        foreach ($cars as $car) {
            $totalExpenses = $car->expenses_sum_amount ?? 0;
            $profitLoss = 0;
            
            if ($car->status === 'sold' && $car->purchase_price && $car->sale_price) {
                $profitLoss = $car->sale_price - $car->purchase_price - $totalExpenses;
            }
            
            $sheet->setCellValue('A' . $row, $car->brand . ' ' . $car->model);
            $sheet->setCellValue('B' . $row, $car->year ?? '-');
            $sheet->setCellValue('C' . $row, $car->plate ?? '-');
            $sheet->setCellValue('D' . $row, $car->purchase_price ?? 0);
            $sheet->setCellValue('E' . $row, $car->sale_price ?? 0);
            $sheet->setCellValue('F' . $row, $totalExpenses);
            $sheet->setCellValue('G' . $row, $profitLoss);
            $sheet->setCellValue('H' . $row, $this->getCarStatusText($car->status));
            $row++;
        }
        
        // Formatação
        $sheet->getStyle('D:G')->getNumberFormat()->setFormatCode('_("R$"* #,##0.00_);_("R$"* $$#,##0.00$$;_("R$"* "-"??_);_(@_)');
        
        foreach (range('A', 'H') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $writer = new Xlsx($spreadsheet);
        $filename = 'relatorio_veiculos_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $writer->save('php://output');
        exit;
    }

    private function getStatusText($status)
    {
        return match($status) {
            'pending' => 'Pendente',
            'paid' => 'Pago',
            'received' => 'Recebido',
            'overdue' => 'Atrasado',
            'cancelled' => 'Cancelado',
            default => 'Indefinido',
        };
    }

    private function getCarStatusText($status)
    {
        return match($status) {
            'owned' => 'Em Estoque',
            'sold' => 'Vendido',
            'maintenance' => 'Manutenção',
            default => 'Indefinido',
        };
    }
}
