<?php

namespace App\Services;

use App\Models\Schedule;
use App\Models\User;
use App\Services\NotificationService;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ScheduleService
{
    protected $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    public function create(array $data): Schedule
    {
        $data['user_id'] = $data['user_id'] ?? Auth::id();
        $data['company_id'] = $data['company_id'] ?? Auth::user()->company_id;

        $schedule = Schedule::create($data);

        // Criar notificações de lembrete
        $this->createReminderNotifications($schedule);

        return $schedule;
    }

    public function update(Schedule $schedule, array $data): Schedule
    {
        $schedule->update($data);

        // Recriar notificações se a data mudou
        if (isset($data['scheduled_at']) || isset($data['reminders'])) {
            $this->createReminderNotifications($schedule);
        }

        return $schedule;
    }

    public function getUpcoming($userId = null, $limit = 10)
    {
        $userId = $userId ?? Auth::id();

        return Schedule::forUser($userId)
            ->upcoming()
            ->orderBy('scheduled_at')
            ->limit($limit)
            ->get();
    }

    public function getToday($userId = null)
    {
        $userId = $userId ?? Auth::id();

        return Schedule::forUser($userId)
            ->today()
            ->orderBy('scheduled_at')
            ->get();
    }

    public function getOverdue($userId = null)
    {
        $userId = $userId ?? Auth::id();

        return Schedule::forUser($userId)
            ->overdue()
            ->orderBy('scheduled_at')
            ->get();
    }

    public function getCalendarEvents($start, $end, $userId = null)
    {
        $userId = $userId ?? Auth::id();

        return Schedule::forUser($userId)
            ->whereBetween('scheduled_at', [$start, $end])
            ->get()
            ->map(function ($schedule) {
                return [
                    'id' => $schedule->id,
                    'title' => $schedule->title,
                    'start' => $schedule->scheduled_at->toISOString(),
                    'end' => $schedule->end_at ? $schedule->end_at->toISOString() : null,
                    'allDay' => $schedule->all_day,
                    'color' => $schedule->color,
                    'className' => 'schedule-' . $schedule->type,
                    'extendedProps' => [
                        'type' => $schedule->type,
                        'priority' => $schedule->priority,
                        'status' => $schedule->status,
                        'description' => $schedule->description,
                        'location' => $schedule->location,
                    ]
                ];
            });
    }

    public function checkReminders($companyId = null)
    {
        $companyId = $companyId ?? Auth::user()->company_id;

        $schedules = Schedule::forCompany($companyId)
            ->pending()
            ->where('scheduled_at', '>', now())
            ->where('scheduled_at', '<=', now()->addHours(2))
            ->whereNotNull('reminders')
            ->get();

        foreach ($schedules as $schedule) {
            foreach ($schedule->reminders as $minutes) {
                if ($schedule->shouldSendReminder($minutes)) {
                    $this->sendReminderNotification($schedule, $minutes);
                }
            }
        }
    }

    protected function createReminderNotifications(Schedule $schedule)
    {
        if (!$schedule->reminders || $schedule->status !== 'pending') {
            return;
        }

        foreach ($schedule->reminders as $minutes) {
            $reminderTime = $schedule->scheduled_at->subMinutes($minutes);
            
            if ($reminderTime > now()) {
                $this->notificationService->createForUser($schedule->user_id, [
                    'title' => 'Lembrete: ' . $schedule->title,
                    'message' => "Seu {$schedule->type_label} está agendado para {$schedule->formatted_date}",
                    'type' => 'info',
                    'priority' => $schedule->priority,
                    'scheduled_for' => $reminderTime,
                    'action_url' => route('schedules.show', $schedule),
                    'action_text' => 'Ver Agenda',
                    'data' => [
                        'schedule_id' => $schedule->id,
                        'reminder_minutes' => $minutes,
                    ],
                ]);
            }
        }
    }

    protected function sendReminderNotification(Schedule $schedule, int $minutes)
    {
        $timeText = $minutes < 60 ? $minutes . ' minutos' : floor($minutes / 60) . ' hora(s)';
        
        $this->notificationService->createForUser($schedule->user_id, [
            'title' => 'Lembrete: ' . $schedule->title,
            'message' => "Seu {$schedule->type_label} começará em {$timeText}",
            'type' => 'warning',
            'priority' => 'high',
            'action_url' => route('schedules.show', $schedule),
            'action_text' => 'Ver Detalhes',
            'data' => [
                'schedule_id' => $schedule->id,
                'reminder_minutes' => $minutes,
            ],
        ]);
    }

    public function createRecurringSchedules(Schedule $schedule)
    {
        if (!$schedule->is_recurring || !$schedule->recurrence_pattern) {
            return [];
        }

        $created = [];
        $currentDate = $schedule->scheduled_at->copy();
        $endDate = $schedule->recurrence_end ?? $currentDate->copy()->addYear();

        while ($currentDate <= $endDate) {
            $currentDate = $this->getNextRecurrenceDate($currentDate, $schedule->recurrence_pattern);
            
            if ($currentDate <= $endDate) {
                $newSchedule = $schedule->replicate();
                $newSchedule->scheduled_at = $currentDate->copy();
                
                if ($schedule->end_at) {
                    $duration = $schedule->scheduled_at->diffInMinutes($schedule->end_at);
                    $newSchedule->end_at = $currentDate->copy()->addMinutes($duration);
                }
                
                $newSchedule->save();
                $created[] = $newSchedule;
            }
        }

        return $created;
    }

    protected function getNextRecurrenceDate(Carbon $date, string $pattern): Carbon
    {
        return match($pattern) {
            'daily' => $date->addDay(),
            'weekly' => $date->addWeek(),
            'monthly' => $date->addMonth(),
            'yearly' => $date->addYear(),
            default => $date->addWeek(),
        };
    }

    public function getStatistics($userId = null, $companyId = null)
    {
        $userId = $userId ?? Auth::id();
        $companyId = $companyId ?? Auth::user()->company_id;

        $query = Schedule::forUser($userId);

        return [
            'total' => $query->count(),
            'pending' => $query->pending()->count(),
            'completed' => $query->completed()->count(),
            'overdue' => $query->overdue()->count(),
            'today' => $query->today()->count(),
            'this_week' => $query->thisWeek()->count(),
            'by_type' => $query->selectRaw('type, count(*) as count')
                              ->groupBy('type')
                              ->pluck('count', 'type')
                              ->toArray(),
            'by_priority' => $query->selectRaw('priority, count(*) as count')
                                  ->groupBy('priority')
                                  ->pluck('count', 'priority')
                                  ->toArray(),
        ];
    }
}
