<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class SettingsService
{
    protected $companyId;
    protected $cachePrefix = 'settings_';

    public function __construct()
    {
        $this->companyId = Auth::user()->company_id ?? null;
    }

    public function get(string $key, $default = null, $companyId = null)
    {
        $companyId = $companyId ?? $this->companyId;
        $cacheKey = $this->cachePrefix . $companyId . '_' . $key;

        return Cache::remember($cacheKey, 3600, function() use ($key, $default, $companyId) {
            $setting = Setting::where('key', $key)
                ->where(function($query) use ($companyId) {
                    $query->where('company_id', $companyId)
                          ->orWhereNull('company_id');
                })
                ->orderBy('company_id', 'desc') // Prioriza configurações da empresa
                ->first();

            return $setting ? $setting->value : $default;
        });
    }

    public function set(string $key, $value, string $type = 'string', $companyId = null)
    {
        $companyId = $companyId ?? $this->companyId;
        
        $setting = Setting::updateOrCreate(
            ['key' => $key, 'company_id' => $companyId],
            [
                'value' => $value,
                'type' => $type,
                'label' => $this->generateLabel($key),
            ]
        );

        // Limpar cache
        $cacheKey = $this->cachePrefix . $companyId . '_' . $key;
        Cache::forget($cacheKey);

        return $setting;
    }

    public function getByGroup(string $group, $companyId = null): array
    {
        $companyId = $companyId ?? $this->companyId;
        
        return Setting::where('group', $group)
            ->where(function($query) use ($companyId) {
                $query->where('company_id', $companyId)
                      ->orWhereNull('company_id');
            })
            ->orderBy('company_id', 'desc')
            ->get()
            ->keyBy('key')
            ->toArray();
    }

    public function getAll($companyId = null): array
    {
        $companyId = $companyId ?? $this->companyId;
        
        return Setting::where(function($query) use ($companyId) {
                $query->where('company_id', $companyId)
                      ->orWhereNull('company_id');
            })
            ->orderBy('group')
            ->orderBy('key')
            ->get()
            ->groupBy('group')
            ->toArray();
    }

    public function initializeDefaults($companyId = null)
    {
        $companyId = $companyId ?? $this->companyId;
        
        $defaults = $this->getDefaultSettings();
        
        foreach ($defaults as $setting) {
            Setting::firstOrCreate(
                [
                    'key' => $setting['key'],
                    'company_id' => $companyId,
                ],
                $setting
            );
        }
    }

    private function getDefaultSettings(): array
    {
        return [
            // Configurações Gerais
            [
                'key' => 'company_name',
                'value' => '',
                'type' => 'string',
                'group' => 'general',
                'label' => 'Nome da Empresa',
                'description' => 'Nome que aparecerá nos relatórios e documentos',
            ],
            [
                'key' => 'company_logo',
                'value' => '',
                'type' => 'string',
                'group' => 'general',
                'label' => 'Logo da Empresa',
                'description' => 'URL ou caminho para o logo da empresa',
            ],
            [
                'key' => 'timezone',
                'value' => 'America/Sao_Paulo',
                'type' => 'string',
                'group' => 'general',
                'label' => 'Fuso Horário',
                'options' => [
                    'America/Sao_Paulo' => 'Brasília (GMT-3)',
                    'America/Manaus' => 'Manaus (GMT-4)',
                    'America/Rio_Branco' => 'Rio Branco (GMT-5)',
                ],
            ],
            [
                'key' => 'date_format',
                'value' => 'd/m/Y',
                'type' => 'string',
                'group' => 'general',
                'label' => 'Formato de Data',
                'options' => [
                    'd/m/Y' => '31/12/2024',
                    'Y-m-d' => '2024-12-31',
                    'd-m-Y' => '31-12-2024',
                ],
            ],
            [
                'key' => 'currency_symbol',
                'value' => 'R$',
                'type' => 'string',
                'group' => 'general',
                'label' => 'Símbolo da Moeda',
            ],

            // Configurações Financeiras
            [
                'key' => 'default_payment_terms',
                'value' => '30',
                'type' => 'integer',
                'group' => 'financial',
                'label' => 'Prazo Padrão de Pagamento (dias)',
                'description' => 'Prazo padrão para vencimento de contas',
            ],
            [
                'key' => 'interest_rate',
                'value' => '1.0',
                'type' => 'string',
                'group' => 'financial',
                'label' => 'Taxa de Juros Mensal (%)',
                'description' => 'Taxa aplicada em contas em atraso',
            ],
            [
                'key' => 'late_fee',
                'value' => '2.0',
                'type' => 'string',
                'group' => 'financial',
                'label' => 'Multa por Atraso (%)',
            ],
            [
                'key' => 'auto_reconcile',
                'value' => true,
                'type' => 'boolean',
                'group' => 'financial',
                'label' => 'Conciliação Automática',
                'description' => 'Tentar conciliar automaticamente transações bancárias',
            ],
            [
                'key' => 'reconcile_tolerance_days',
                'value' => '5',
                'type' => 'integer',
                'group' => 'financial',
                'label' => 'Tolerância para Conciliação (dias)',
                'description' => 'Diferença máxima de dias para conciliação automática',
            ],

            // Notificações
            [
                'key' => 'notify_overdue',
                'value' => true,
                'type' => 'boolean',
                'group' => 'notifications',
                'label' => 'Notificar Contas Vencidas',
            ],
            [
                'key' => 'notify_due_soon',
                'value' => true,
                'type' => 'boolean',
                'group' => 'notifications',
                'label' => 'Notificar Vencimentos Próximos',
            ],
            [
                'key' => 'due_soon_days',
                'value' => '7',
                'type' => 'integer',
                'group' => 'notifications',
                'label' => 'Dias para Vencimento Próximo',
            ],
            [
                'key' => 'email_notifications',
                'value' => false,
                'type' => 'boolean',
                'group' => 'notifications',
                'label' => 'Enviar Notificações por E-mail',
            ],
            [
                'key' => 'notification_email',
                'value' => '',
                'type' => 'string',
                'group' => 'notifications',
                'label' => 'E-mail para Notificações',
            ],

            // Backup
            [
                'key' => 'auto_backup',
                'value' => false,
                'type' => 'boolean',
                'group' => 'backup',
                'label' => 'Backup Automático',
            ],
            [
                'key' => 'backup_frequency',
                'value' => 'weekly',
                'type' => 'string',
                'group' => 'backup',
                'label' => 'Frequência do Backup',
                'options' => [
                    'daily' => 'Diário',
                    'weekly' => 'Semanal',
                    'monthly' => 'Mensal',
                ],
            ],
            [
                'key' => 'backup_retention_days',
                'value' => '30',
                'type' => 'integer',
                'group' => 'backup',
                'label' => 'Retenção de Backups (dias)',
            ],

            // Relatórios
            [
                'key' => 'default_report_format',
                'value' => 'pdf',
                'type' => 'string',
                'group' => 'reports',
                'label' => 'Formato Padrão de Relatórios',
                'options' => [
                    'pdf' => 'PDF',
                    'excel' => 'Excel',
                    'csv' => 'CSV',
                ],
            ],
            [
                'key' => 'include_logo_reports',
                'value' => true,
                'type' => 'boolean',
                'group' => 'reports',
                'label' => 'Incluir Logo nos Relatórios',
            ],
        ];
    }

    private function generateLabel(string $key): string
    {
        return ucwords(str_replace(['_', '-'], ' ', $key));
    }

    public function clearCache($companyId = null)
    {
        $companyId = $companyId ?? $this->companyId;
        $pattern = $this->cachePrefix . $companyId . '_*';
        
        // Limpar cache com padrão específico
        $keys = Cache::getRedis()->keys($pattern);
        if (!empty($keys)) {
            Cache::getRedis()->del($keys);
        }
    }
}
