<?php

/**
 * Script de verificação do sistema
 * Verifica se todos os requisitos estão atendidos
 */

echo "=== Verificação do Sistema ===\n\n";

$checks = [];

// Verificar versão do PHP
$phpVersion = PHP_VERSION;
$checks['PHP Version'] = [
    'status' => version_compare($phpVersion, '8.1.0', '>='),
    'message' => "PHP {$phpVersion} " . (version_compare($phpVersion, '8.1.0', '>=') ? '✅' : '❌ (Requer 8.1+)')
];

// Verificar extensões PHP
$requiredExtensions = [
    'pdo', 'pdo_mysql', 'mbstring', 'tokenizer', 'xml', 'ctype', 
    'json', 'bcmath', 'fileinfo', 'openssl', 'zip', 'gd'
];

foreach ($requiredExtensions as $ext) {
    $checks["Extensão {$ext}"] = [
        'status' => extension_loaded($ext),
        'message' => extension_loaded($ext) ? '✅' : '❌'
    ];
}

// Verificar permissões de diretórios
$directories = [
    'storage/app' => 775,
    'storage/framework' => 775,
    'storage/logs' => 775,
    'bootstrap/cache' => 775
];

foreach ($directories as $dir => $requiredPerm) {
    $path = __DIR__ . '/../' . $dir;
    if (is_dir($path)) {
        $perms = substr(sprintf('%o', fileperms($path)), -3);
        $checks["Permissão {$dir}"] = [
            'status' => $perms >= $requiredPerm,
            'message' => "Atual: {$perms} " . ($perms >= $requiredPerm ? '✅' : "❌ (Requer {$requiredPerm})")
        ];
    } else {
        $checks["Diretório {$dir}"] = [
            'status' => false,
            'message' => '❌ Não existe'
        ];
    }
}

// Verificar arquivo .env
$envPath = __DIR__ . '/../.env';
$checks['Arquivo .env'] = [
    'status' => file_exists($envPath),
    'message' => file_exists($envPath) ? '✅' : '❌ Não encontrado'
];

// Verificar se o Composer foi executado
$vendorPath = __DIR__ . '/../vendor';
$checks['Dependências Composer'] = [
    'status' => is_dir($vendorPath) && file_exists($vendorPath . '/autoload.php'),
    'message' => (is_dir($vendorPath) && file_exists($vendorPath . '/autoload.php')) ? '✅' : '❌ Execute: composer install'
];

// Exibir resultados
$allPassed = true;
foreach ($checks as $name => $check) {
    echo sprintf("%-25s: %s\n", $name, $check['message']);
    if (!$check['status']) {
        $allPassed = false;
    }
}

echo "\n";
if ($allPassed) {
    echo "🎉 Todos os requisitos foram atendidos!\n";
    echo "O sistema está pronto para ser usado.\n";
} else {
    echo "⚠️  Alguns requisitos não foram atendidos.\n";
    echo "Corrija os problemas antes de continuar.\n";
}

echo "\n=== Fim da Verificação ===\n";
