import "./bootstrap"
import bootstrap from "bootstrap"

// Sistema de Gestão Financeira - JavaScript Principal

document.addEventListener("DOMContentLoaded", () => {
  // Inicializar componentes
  initializeComponents()

  // Configurar eventos globais
  setupGlobalEvents()

  // Configurar formulários
  setupForms()

  // Configurar tabelas
  setupTables()
})

function initializeComponents() {
  // Inicializar tooltips
  const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
  tooltipTriggerList.map((tooltipTriggerEl) => new bootstrap.Tooltip(tooltipTriggerEl))

  // Inicializar popovers
  const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'))
  popoverTriggerList.map((popoverTriggerEl) => new bootstrap.Popover(popoverTriggerEl))

  // Inicializar máscaras de input
  setupInputMasks()

  // Inicializar validação de formulários
  setupFormValidation()
}

function setupGlobalEvents() {
  // Confirmação de exclusão
  document.addEventListener("click", (e) => {
    if (e.target.matches(".btn-delete, .delete-btn")) {
      e.preventDefault()
      const message = e.target.getAttribute("data-message") || "Tem certeza que deseja excluir este item?"
      if (confirm(message)) {
        if (e.target.closest("form")) {
          e.target.closest("form").submit()
        } else if (e.target.href) {
          window.location.href = e.target.href
        }
      }
    }
  })

  // Loading em formulários
  document.addEventListener("submit", (e) => {
    if (e.target.matches("form:not(.no-loading)")) {
      window.showLoading()
    }
  })

  // Auto-save em formulários (opcional)
  setupAutoSave()
}

function setupForms() {
  // Formatação de moeda
  const currencyInputs = document.querySelectorAll(".currency-input")
  currencyInputs.forEach((input) => {
    input.addEventListener("input", function () {
      formatCurrency(this)
    })
  })

  // Formatação de CPF/CNPJ
  const documentInputs = document.querySelectorAll(".document-input")
  documentInputs.forEach((input) => {
    input.addEventListener("input", function () {
      formatDocument(this)
    })
  })

  // Formatação de telefone
  const phoneInputs = document.querySelectorAll(".phone-input")
  phoneInputs.forEach((input) => {
    input.addEventListener("input", function () {
      formatPhone(this)
    })
  })

  // Formatação de CEP
  const cepInputs = document.querySelectorAll(".cep-input")
  cepInputs.forEach((input) => {
    input.addEventListener("input", function () {
      formatCEP(this)
    })

    input.addEventListener("blur", function () {
      if (this.value.length === 9) {
        searchCEP(this.value)
      }
    })
  })
}

function setupTables() {
  // Seleção múltipla em tabelas
  const selectAllCheckbox = document.querySelector("#select-all")
  if (selectAllCheckbox) {
    selectAllCheckbox.addEventListener("change", () => {
      const checkboxes = document.querySelectorAll(".row-checkbox")
      checkboxes.forEach((checkbox) => {
        checkbox.checked = selectAllCheckbox.checked
      })
      updateBulkActions()
    })
  }

  // Checkboxes individuais
  const rowCheckboxes = document.querySelectorAll(".row-checkbox")
  rowCheckboxes.forEach((checkbox) => {
    checkbox.addEventListener("change", () => {
      updateBulkActions()
    })
  })

  // Ordenação de tabelas
  setupTableSorting()
}

function setupInputMasks() {
  // Implementar máscaras personalizadas se necessário
}

function setupFormValidation() {
  // Validação customizada de formulários
  const forms = document.querySelectorAll(".needs-validation")
  forms.forEach((form) => {
    form.addEventListener("submit", (event) => {
      if (!form.checkValidity()) {
        event.preventDefault()
        event.stopPropagation()
      }
      form.classList.add("was-validated")
    })
  })
}

function setupAutoSave() {
  // Auto-save para formulários longos (opcional)
  const autoSaveForms = document.querySelectorAll(".auto-save")
  autoSaveForms.forEach((form) => {
    const inputs = form.querySelectorAll("input, textarea, select")
    inputs.forEach((input) => {
      input.addEventListener("change", () => {
        debounce(() => {
          autoSaveForm(form)
        }, 2000)()
      })
    })
  })
}

function setupTableSorting() {
  const sortableHeaders = document.querySelectorAll(".sortable")
  sortableHeaders.forEach((header) => {
    header.addEventListener("click", function () {
      const column = this.getAttribute("data-column")
      const currentSort = this.getAttribute("data-sort") || "asc"
      const newSort = currentSort === "asc" ? "desc" : "asc"

      // Atualizar URL com parâmetros de ordenação
      const url = new URL(window.location)
      url.searchParams.set("sort", column)
      url.searchParams.set("direction", newSort)
      window.location.href = url.toString()
    })
  })
}

// Funções utilitárias
function formatCurrency(input) {
  let value = input.value.replace(/\D/g, "")
  value = (value / 100).toFixed(2)
  value = value.replace(".", ",")
  value = value.replace(/\B(?=(\d{3})+(?!\d))/g, ".")
  input.value = "R$ " + value
}

function formatDocument(input) {
  let value = input.value.replace(/\D/g, "")

  if (value.length <= 11) {
    // CPF
    value = value.replace(/(\d{3})(\d)/, "$1.$2")
    value = value.replace(/(\d{3})(\d)/, "$1.$2")
    value = value.replace(/(\d{3})(\d{1,2})$/, "$1-$2")
  } else {
    // CNPJ
    value = value.replace(/^(\d{2})(\d)/, "$1.$2")
    value = value.replace(/^(\d{2})\.(\d{3})(\d)/, "$1.$2.$3")
    value = value.replace(/\.(\d{3})(\d)/, ".$1/$2")
    value = value.replace(/(\d{4})(\d)/, "$1-$2")
  }

  input.value = value
}

function formatPhone(input) {
  let value = input.value.replace(/\D/g, "")

  if (value.length <= 10) {
    value = value.replace(/(\d{2})(\d)/, "($1) $2")
    value = value.replace(/(\d{4})(\d)/, "$1-$2")
  } else {
    value = value.replace(/(\d{2})(\d)/, "($1) $2")
    value = value.replace(/(\d{5})(\d)/, "$1-$2")
  }

  input.value = value
}

function formatCEP(input) {
  let value = input.value.replace(/\D/g, "")
  value = value.replace(/(\d{5})(\d)/, "$1-$2")
  input.value = value
}

function searchCEP(cep) {
  cep = cep.replace(/\D/g, "")

  if (cep.length === 8) {
    fetch(`https://viacep.com.br/ws/${cep}/json/`)
      .then((response) => response.json())
      .then((data) => {
        if (!data.erro) {
          fillAddressFields(data)
        }
      })
      .catch((error) => {
        console.error("Erro ao buscar CEP:", error)
      })
  }
}

function fillAddressFields(data) {
  const fields = {
    address: data.logradouro,
    neighborhood: data.bairro,
    city: data.localidade,
    state: data.uf,
  }

  Object.keys(fields).forEach((field) => {
    const input = document.querySelector(`[name="${field}"]`)
    if (input && fields[field]) {
      input.value = fields[field]
    }
  })
}

function updateBulkActions() {
  const checkedBoxes = document.querySelectorAll(".row-checkbox:checked")
  const bulkActions = document.querySelector(".bulk-actions")

  if (bulkActions) {
    if (checkedBoxes.length > 0) {
      bulkActions.style.display = "block"
      bulkActions.querySelector(".selected-count").textContent = checkedBoxes.length
    } else {
      bulkActions.style.display = "none"
    }
  }
}

function autoSaveForm(form) {
  const formData = new FormData(form)
  const url = form.getAttribute("data-autosave-url") || form.action

  fetch(url, {
    method: "POST",
    body: formData,
    headers: {
      "X-CSRF-TOKEN": document.querySelector('meta[name="csrf-token"]').getAttribute("content"),
      "X-Auto-Save": "true",
    },
  })
    .then((response) => response.json())
    .then((data) => {
      if (data.success) {
        window.showNotification("Dados salvos automaticamente", "info")
      }
    })
    .catch((error) => {
      console.error("Erro no auto-save:", error)
    })
}

function debounce(func, wait) {
  let timeout
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout)
      func(...args)
    }
    clearTimeout(timeout)
    timeout = setTimeout(later, wait)
  }
}

// Funções globais
function showLoading() {
  // Implementar lógica de showLoading
}

function hideLoading() {
  // Implementar lógica de hideLoading
}

function showNotification(message, type) {
  // Implementar lógica de showNotification
}

function confirmDelete(message) {
  // Implementar lógica de confirmDelete
}

// Exportar funções globais
window.formatCurrency = formatCurrency
window.showLoading = showLoading
window.hideLoading = hideLoading
window.showNotification = showNotification
window.confirmDelete = confirmDelete
