@extends('layouts.app')

@section('title', 'Biblioteca de Categorias')
@section('page-title', 'Biblioteca de Categorias')

@section('content')
<div class="row mb-4">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <h5 class="card-title">
                    <i class="bi bi-collection"></i> Biblioteca de Categorias
                </h5>
                <p class="card-text text-muted">
                    Escolha entre mais de 150 categorias pré-definidas para organizar suas finanças.
                </p>
                
                <!-- Estatísticas -->
                <div class="row text-center mb-3">
                    <div class="col-3">
                        <div class="border-end">
                            <h4 class="text-primary mb-0">{{ $statistics['total_available'] }}</h4>
                            <small class="text-muted">Disponíveis</small>
                        </div>
                    </div>
                    <div class="col-3">
                        <div class="border-end">
                            <h4 class="text-success mb-0">{{ $statistics['imported'] }}</h4>
                            <small class="text-muted">Importadas</small>
                        </div>
                    </div>
                    <div class="col-3">
                        <div class="border-end">
                            <h4 class="text-info mb-0">{{ $statistics['custom'] }}</h4>
                            <small class="text-muted">Personalizadas</small>
                        </div>
                    </div>
                    <div class="col-3">
                        <h4 class="text-warning mb-0">{{ $statistics['import_percentage'] }}%</h4>
                        <small class="text-muted">Cobertura</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-body">
                <h6 class="card-title">Ações Rápidas</h6>
                <div class="d-grid gap-2">
                    <form method="POST" action="{{ route('categories.import-popular') }}">
                        @csrf
                        <button type="submit" class="btn btn-success btn-sm w-100">
                            <i class="bi bi-star"></i> Importar Populares
                        </button>
                    </form>
                    
                    <form method="POST" action="{{ route('categories.initialize-defaults') }}">
                        @csrf
                        <button type="submit" class="btn btn-primary btn-sm w-100">
                            <i class="bi bi-gear"></i> Categorias Essenciais
                        </button>
                    </form>
                    
                    <a href="{{ route('categories.index') }}" class="btn btn-outline-secondary btn-sm">
                        <i class="bi bi-arrow-left"></i> Voltar às Categorias
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Buscar</label>
                <input type="text" name="search" class="form-control" 
                       value="{{ $filters['search'] ?? '' }}" 
                       placeholder="Nome ou palavra-chave...">
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Tipo</label>
                <select name="type" class="form-select">
                    <option value="">Todos</option>
                    <option value="income" {{ ($filters['type'] ?? '') === 'income' ? 'selected' : '' }}>Receita</option>
                    <option value="expense" {{ ($filters['type'] ?? '') === 'expense' ? 'selected' : '' }}>Despesa</option>
                </select>
            </div>
            
            <div class="col-md-3">
                <label class="form-label">Grupo</label>
                <select name="group" class="form-select">
                    <option value="">Todos os grupos</option>
                    @foreach($groups as $group)
                        <option value="{{ $group }}" {{ ($filters['group'] ?? '') === $group ? 'selected' : '' }}>
                            {{ $group }}
                        </option>
                    @endforeach
                </select>
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Filtro</label>
                <div class="form-check">
                    <input type="checkbox" name="popular" value="1" class="form-check-input" 
                           {{ isset($filters['popular']) ? 'checked' : '' }}>
                    <label class="form-check-label">Apenas populares</label>
                </div>
            </div>
            
            <div class="col-md-2">
                <label class="form-label">&nbsp;</label>
                <div class="d-grid">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-search"></i> Filtrar
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Formulário para importação múltipla -->
<form id="importForm" method="POST" action="{{ route('categories.import-selected') }}">
    @csrf
    <div class="d-flex justify-content-between align-items-center mb-3">
        <div>
            <button type="button" id="selectAll" class="btn btn-outline-primary btn-sm">
                <i class="bi bi-check-all"></i> Selecionar Todos
            </button>
            <button type="button" id="selectNone" class="btn btn-outline-secondary btn-sm">
                <i class="bi bi-x"></i> Limpar Seleção
            </button>
        </div>
        
        <div>
            <span id="selectedCount" class="badge bg-info me-2">0 selecionadas</span>
            <button type="submit" id="importSelected" class="btn btn-success" disabled>
                <i class="bi bi-download"></i> Importar Selecionadas
            </button>
        </div>
    </div>

    <!-- Categorias por Grupo -->
    @foreach($categoriesByGroup as $group => $categories)
        <div class="card mb-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">
                    <i class="bi bi-folder"></i> {{ $group ?: 'Sem Grupo' }}
                    <span class="badge bg-secondary">{{ $categories->count() }}</span>
                </h5>
                
                @if($group)
                    <form method="POST" action="{{ route('categories.import-group') }}" class="d-inline">
                        @csrf
                        <input type="hidden" name="group" value="{{ $group }}">
                        <button type="submit" class="btn btn-outline-success btn-sm">
                            <i class="bi bi-download"></i> Importar Grupo
                        </button>
                    </form>
                @endif
            </div>
            
            <div class="card-body">
                <div class="row">
                    @foreach($categories as $category)
                        <div class="col-md-6 col-lg-4 mb-3">
                            <div class="card h-100 category-card {{ in_array($category->id, $imported) ? 'border-success' : '' }}">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between align-items-start mb-2">
                                        <div class="form-check">
                                            <input type="checkbox" name="categories[]" value="{{ $category->id }}" 
                                                   class="form-check-input category-checkbox"
                                                   {{ in_array($category->id, $imported) ? 'disabled' : '' }}>
                                        </div>
                                        
                                        <div class="text-end">
                                            @if($category->is_popular)
                                                <span class="badge bg-warning text-dark">
                                                    <i class="bi bi-star"></i> Popular
                                                </span>
                                            @endif
                                            
                                            @if(in_array($category->id, $imported))
                                                <span class="badge bg-success">
                                                    <i class="bi bi-check"></i> Importada
                                                </span>
                                            @endif
                                        </div>
                                    </div>
                                    
                                    <div class="d-flex align-items-center mb-2">
                                        <span class="me-2" style="color: {{ $category->color }}; font-size: 1.2em;">
                                            <i class="{{ $category->icon }}"></i>
                                        </span>
                                        <h6 class="card-title mb-0">{{ $category->name }}</h6>
                                    </div>
                                    
                                    <p class="card-text text-muted small">{{ $category->description }}</p>
                                    
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="badge {{ $category->type === 'income' ? 'bg-success' : 'bg-danger' }}">
                                            {{ $category->type === 'income' ? 'Receita' : 'Despesa' }}
                                        </span>
                                        
                                        @if($category->usage_count > 0)
                                            <small class="text-muted">
                                                <i class="bi bi-graph-up"></i> {{ $category->usage_count }} usos
                                            </small>
                                        @endif
                                    </div>
                                    
                                    @if($category->keywords)
                                        <div class="mt-2">
                                            <small class="text-muted">
                                                <i class="bi bi-tags"></i> {{ $category->keywords_string }}
                                            </small>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
            </div>
        </div>
    @endforeach
</form>

@if($categoriesByGroup->isEmpty())
    <div class="text-center py-5">
        <i class="bi bi-search display-1 text-muted"></i>
        <h4 class="text-muted">Nenhuma categoria encontrada</h4>
        <p class="text-muted">Tente ajustar os filtros de busca.</p>
        <a href="{{ route('categories.library') }}" class="btn btn-primary">
            <i class="bi bi-arrow-clockwise"></i> Limpar Filtros
        </a>
    </div>
@endif
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const checkboxes = document.querySelectorAll('.category-checkbox:not(:disabled)');
    const selectAllBtn = document.getElementById('selectAll');
    const selectNoneBtn = document.getElementById('selectNone');
    const selectedCount = document.getElementById('selectedCount');
    const importBtn = document.getElementById('importSelected');
    
    function updateSelectedCount() {
        const selected = document.querySelectorAll('.category-checkbox:checked').length;
        selectedCount.textContent = selected + ' selecionadas';
        importBtn.disabled = selected === 0;
    }
    
    selectAllBtn.addEventListener('click', function() {
        checkboxes.forEach(cb => cb.checked = true);
        updateSelectedCount();
    });
    
    selectNoneBtn.addEventListener('click', function() {
        checkboxes.forEach(cb => cb.checked = false);
        updateSelectedCount();
    });
    
    checkboxes.forEach(cb => {
        cb.addEventListener('change', updateSelectedCount);
    });
    
    updateSelectedCount();
});
</script>
@endpush

@push('styles')
<style>
.category-card {
    transition: all 0.3s ease;
    cursor: pointer;
}

.category-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.category-card.border-success {
    background-color: rgba(25, 135, 84, 0.05);
}
</style>
@endpush
