@extends('layouts.app')

@section('title', 'Calendário')
@section('page-title', 'Calendário')

@push('styles')
<link href='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.css' rel='stylesheet' />
<style>
.fc-event {
    border: none !important;
    border-radius: 4px !important;
}
.fc-daygrid-event {
    font-size: 12px;
}
.calendar-legend {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    margin-bottom: 20px;
}
.legend-item {
    display: flex;
    align-items: center;
    gap: 5px;
}
.legend-color {
    width: 12px;
    height: 12px;
    border-radius: 2px;
}
</style>
@endpush

@section('content')
<div class="row mb-4">
    <!-- Estatísticas Rápidas -->
    <div class="col-md-3">
        <div class="card stats-card">
            <div class="card-body text-center">
                <i class="bi bi-calendar-day fs-1"></i>
                <h3 class="mt-2">{{ $statistics['today'] }}</h3>
                <p class="mb-0">Hoje</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body text-center">
                <i class="bi bi-calendar-week fs-1"></i>
                <h3 class="mt-2">{{ $statistics['this_week'] }}</h3>
                <p class="mb-0">Esta Semana</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body text-center">
                <i class="bi bi-clock fs-1"></i>
                <h3 class="mt-2">{{ $statistics['pending'] }}</h3>
                <p class="mb-0">Pendentes</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-danger text-white">
            <div class="card-body text-center">
                <i class="bi bi-exclamation-triangle fs-1"></i>
                <h3 class="mt-2">{{ $statistics['overdue'] }}</h3>
                <p class="mb-0">Em Atraso</p>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">
            <i class="bi bi-calendar3"></i> Calendário
        </h5>
        <div>
            <a href="{{ route('schedules.index') }}" class="btn btn-outline-primary btn-sm">
                <i class="bi bi-list-ul"></i> Lista
            </a>
            <a href="{{ route('schedules.create') }}" class="btn btn-primary btn-sm">
                <i class="bi bi-plus"></i> Novo Evento
            </a>
        </div>
    </div>
    <div class="card-body">
        <!-- Legenda -->
        <div class="calendar-legend">
            <div class="legend-item">
                <div class="legend-color" style="background-color: #0d6efd;"></div>
                <small>Compromisso</small>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color: #198754;"></div>
                <small>Tarefa</small>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color: #ffc107;"></div>
                <small>Lembrete</small>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color: #6f42c1;"></div>
                <small>Reunião</small>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color: #dc3545;"></div>
                <small>Prazo</small>
            </div>
        </div>

        <!-- Calendário -->
        <div id="calendar"></div>
    </div>
</div>

<!-- Modal para Detalhes do Evento -->
<div class="modal fade" id="eventModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="eventModalTitle"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="eventModalBody">
                <!-- Conteúdo será preenchido via JavaScript -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                <a href="#" id="eventEditBtn" class="btn btn-primary">Editar</a>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.js'></script>
<script src='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/locales/pt-br.global.min.js'></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var calendarEl = document.getElementById('calendar');
    var calendar = new FullCalendar.Calendar(calendarEl, {
        locale: 'pt-br',
        initialView: 'dayGridMonth',
        headerToolbar: {
            left: 'prev,next today',
            center: 'title',
            right: 'dayGridMonth,timeGridWeek,timeGridDay'
        },
        height: 'auto',
        events: function(fetchInfo, successCallback, failureCallback) {
            fetch('/schedules/calendar?' + new URLSearchParams({
                start: fetchInfo.startStr,
                end: fetchInfo.endStr
            }))
            .then(response => response.json())
            .then(data => successCallback(data))
            .catch(error => failureCallback(error));
        },
        eventClick: function(info) {
            showEventModal(info.event);
        },
        dateClick: function(info) {
            // Redirecionar para criar evento na data clicada
            window.location.href = '/schedules/create?date=' + info.dateStr;
        },
        eventDidMount: function(info) {
            // Adicionar tooltip
            info.el.setAttribute('title', info.event.extendedProps.description || info.event.title);
        }
    });
    
    calendar.render();

    function showEventModal(event) {
        const modal = new bootstrap.Modal(document.getElementById('eventModal'));
        const props = event.extendedProps;
        
        document.getElementById('eventModalTitle').textContent = event.title;
        document.getElementById('eventEditBtn').href = '/schedules/' + event.id + '/edit';
        
        let modalBody = `
            <div class="row">
                <div class="col-sm-4"><strong>Tipo:</strong></div>
                <div class="col-sm-8">${getTypeLabel(props.type)}</div>
            </div>
            <div class="row mt-2">
                <div class="col-sm-4"><strong>Data/Hora:</strong></div>
                <div class="col-sm-8">${formatDateTime(event.start, event.end, event.allDay)}</div>
            </div>
            <div class="row mt-2">
                <div class="col-sm-4"><strong>Prioridade:</strong></div>
                <div class="col-sm-8"><span class="badge ${getPriorityBadge(props.priority)}">${getPriorityLabel(props.priority)}</span></div>
            </div>
            <div class="row mt-2">
                <div class="col-sm-4"><strong>Status:</strong></div>
                <div class="col-sm-8"><span class="badge ${getStatusBadge(props.status)}">${getStatusLabel(props.status)}</span></div>
            </div>
        `;
        
        if (props.description) {
            modalBody += `
                <div class="row mt-2">
                    <div class="col-sm-4"><strong>Descrição:</strong></div>
                    <div class="col-sm-8">${props.description}</div>
                </div>
            `;
        }
        
        if (props.location) {
            modalBody += `
                <div class="row mt-2">
                    <div class="col-sm-4"><strong>Local:</strong></div>
                    <div class="col-sm-8">${props.location}</div>
                </div>
            `;
        }
        
        document.getElementById('eventModalBody').innerHTML = modalBody;
        modal.show();
    }

    function getTypeLabel(type) {
        const types = {
            'appointment': 'Compromisso',
            'reminder': 'Lembrete',
            'task': 'Tarefa',
            'meeting': 'Reunião',
            'deadline': 'Prazo'
        };
        return types[type] || type;
    }

    function getPriorityLabel(priority) {
        const priorities = {
            'low': 'Baixa',
            'medium': 'Média',
            'high': 'Alta',
            'urgent': 'Urgente'
        };
        return priorities[priority] || priority;
    }

    function getPriorityBadge(priority) {
        const badges = {
            'low': 'bg-secondary',
            'medium': 'bg-primary',
            'high': 'bg-warning',
            'urgent': 'bg-danger'
        };
        return badges[priority] || 'bg-secondary';
    }

    function getStatusLabel(status) {
        const statuses = {
            'pending': 'Pendente',
            'completed': 'Concluído',
            'cancelled': 'Cancelado'
        };
        return statuses[status] || status;
    }

    function getStatusBadge(status) {
        const badges = {
            'pending': 'bg-warning',
            'completed': 'bg-success',
            'cancelled': 'bg-secondary'
        };
        return badges[status] || 'bg-secondary';
    }

    function formatDateTime(start, end, allDay) {
        const startDate = new Date(start);
        let formatted = startDate.toLocaleDateString('pt-BR');
        
        if (!allDay) {
            formatted += ' às ' + startDate.toLocaleTimeString('pt-BR', {hour: '2-digit', minute: '2-digit'});
            
            if (end) {
                const endDate = new Date(end);
                formatted += ' - ' + endDate.toLocaleTimeString('pt-BR', {hour: '2-digit', minute: '2-digit'});
            }
        } else {
            formatted += ' (Dia inteiro)';
        }
        
        return formatted;
    }
});
</script>
@endpush
