@extends('layouts.app')

@section('title', isset($schedule) ? 'Editar Evento' : 'Novo Evento')
@section('page-title', isset($schedule) ? 'Editar Evento' : 'Novo Evento')

@push('styles')
<style>
.color-picker {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}
.color-option {
    width: 30px;
    height: 30px;
    border-radius: 50%;
    border: 2px solid transparent;
    cursor: pointer;
    transition: all 0.2s;
}
.color-option:hover,
.color-option.selected {
    border-color: #000;
    transform: scale(1.1);
}
.reminder-tags {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    margin-top: 10px;
}
.reminder-tag {
    background: #e9ecef;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 12px;
    display: flex;
    align-items: center;
    gap: 5px;
}
.reminder-tag .remove {
    cursor: pointer;
    color: #dc3545;
}
</style>
@endpush

@section('content')
<div class="row justify-content-center">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-calendar-plus"></i> 
                    {{ isset($schedule) ? 'Editar Evento' : 'Novo Evento' }}
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ isset($schedule) ? route('schedules.update', $schedule) : route('schedules.store') }}">
                    @csrf
                    @if(isset($schedule))
                        @method('PUT')
                    @endif

                    <div class="row">
                        <div class="col-md-8">
                            <!-- Título -->
                            <div class="mb-3">
                                <label for="title" class="form-label">Título *</label>
                                <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                       id="title" name="title" 
                                       value="{{ old('title', $schedule->title ?? '') }}" required>
                                @error('title')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Descrição -->
                            <div class="mb-3">
                                <label for="description" class="form-label">Descrição</label>
                                <textarea class="form-control @error('description') is-invalid @enderror" 
                                          id="description" name="description" rows="3">{{ old('description', $schedule->description ?? '') }}</textarea>
                                @error('description')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-4">
                            <!-- Tipo -->
                            <div class="mb-3">
                                <label for="type" class="form-label">Tipo *</label>
                                <select class="form-select @error('type') is-invalid @enderror" id="type" name="type" required>
                                    <option value="">Selecione...</option>
                                    <option value="appointment" {{ old('type', $schedule->type ?? '') == 'appointment' ? 'selected' : '' }}>Compromisso</option>
                                    <option value="reminder" {{ old('type', $schedule->type ?? '') == 'reminder' ? 'selected' : '' }}>Lembrete</option>
                                    <option value="task" {{ old('type', $schedule->type ?? '') == 'task' ? 'selected' : '' }}>Tarefa</option>
                                    <option value="meeting" {{ old('type', $schedule->type ?? '') == 'meeting' ? 'selected' : '' }}>Reunião</option>
                                    <option value="deadline" {{ old('type', $schedule->type ?? '') == 'deadline' ? 'selected' : '' }}>Prazo</option>
                                </select>
                                @error('type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Prioridade -->
                            <div class="mb-3">
                                <label for="priority" class="form-label">Prioridade *</label>
                                <select class="form-select @error('priority') is-invalid @enderror" id="priority" name="priority" required>
                                    <option value="low" {{ old('priority', $schedule->priority ?? 'medium') == 'low' ? 'selected' : '' }}>Baixa</option>
                                    <option value="medium" {{ old('priority', $schedule->priority ?? 'medium') == 'medium' ? 'selected' : '' }}>Média</option>
                                    <option value="high" {{ old('priority', $schedule->priority ?? 'medium') == 'high' ? 'selected' : '' }}>Alta</option>
                                    <option value="urgent" {{ old('priority', $schedule->priority ?? 'medium') == 'urgent' ? 'selected' : '' }}>Urgente</option>
                                </select>
                                @error('priority')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <!-- Data e Hora de Início -->
                            <div class="mb-3">
                                <label for="scheduled_at" class="form-label">Data e Hora de Início *</label>
                                <input type="datetime-local" class="form-control @error('scheduled_at') is-invalid @enderror" 
                                       id="scheduled_at" name="scheduled_at" 
                                       value="{{ old('scheduled_at', isset($schedule) ? $schedule->scheduled_at->format('Y-m-d\TH:i') : (request('date') ? request('date') . 'T09:00' : '')) }}" required>
                                @error('scheduled_at')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <!-- Data e Hora de Término -->
                            <div class="mb-3">
                                <label for="end_at" class="form-label">Data e Hora de Término</label>
                                <input type="datetime-local" class="form-control @error('end_at') is-invalid @enderror" 
                                       id="end_at" name="end_at" 
                                       value="{{ old('end_at', isset($schedule) && $schedule->end_at ? $schedule->end_at->format('Y-m-d\TH:i') : '') }}">
                                @error('end_at')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <!-- Dia Inteiro -->
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="all_day" name="all_day" value="1"
                                   {{ old('all_day', $schedule->all_day ?? false) ? 'checked' : '' }}>
                            <label class="form-check-label" for="all_day">
                                Evento de dia inteiro
                            </label>
                        </div>
                    </div>

                    <!-- Local -->
                    <div class="mb-3">
                        <label for="location" class="form-label">Local</label>
                        <input type="text" class="form-control @error('location') is-invalid @enderror" 
                               id="location" name="location" 
                               value="{{ old('location', $schedule->location ?? '') }}">
                        @error('location')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- Participantes -->
                    <div class="mb-3">
                        <label for="participants_string" class="form-label">Participantes</label>
                        <input type="text" class="form-control" id="participants_string" name="participants_string" 
                               value="{{ old('participants_string', isset($schedule) && $schedule->participants ? implode(', ', $schedule->participants) : '') }}"
                               placeholder="Separe os nomes por vírgula">
                        <small class="form-text text-muted">Exemplo: João Silva, Maria Santos, Pedro Costa</small>
                    </div>

                    <!-- Lembretes -->
                    <div class="mb-3">
                        <label class="form-label">Lembretes</label>
                        <div class="input-group">
                            <input type="number" class="form-control" id="reminder_input" placeholder="Minutos antes" min="1" max="10080">
                            <button type="button" class="btn btn-outline-secondary" onclick="addReminder()">
                                <i class="bi bi-plus"></i> Adicionar
                            </button>
                        </div>
                        <div class="reminder-tags" id="reminder_tags">
                            @if(isset($schedule) && $schedule->reminders)
                                @foreach($schedule->reminders as $reminder)
                                    <span class="reminder-tag">
                                        {{ $reminder }} min
                                        <span class="remove" onclick="removeReminder(this, {{ $reminder }})">&times;</span>
                                    </span>
                                @endforeach
                            @endif
                        </div>
                        <input type="hidden" name="reminders_string" id="reminders_string" 
                               value="{{ old('reminders_string', isset($schedule) && $schedule->reminders ? implode(',', $schedule->reminders) : '') }}">
                    </div>

                    <!-- Cor -->
                    <div class="mb-3">
                        <label class="form-label">Cor</label>
                        <div class="color-picker">
                            @php
                                $colors = ['#0d6efd', '#198754', '#ffc107', '#dc3545', '#6f42c1', '#fd7e14', '#20c997', '#6c757d'];
                                $selectedColor = old('color', $schedule->color ?? '#0d6efd');
                            @endphp
                            @foreach($colors as $color)
                                <div class="color-option {{ $selectedColor == $color ? 'selected' : '' }}" 
                                     style="background-color: {{ $color }}" 
                                     onclick="selectColor('{{ $color }}')"></div>
                            @endforeach
                        </div>
                        <input type="hidden" name="color" id="color" value="{{ $selectedColor }}">
                    </div>

                    <!-- Recorrência -->
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="is_recurring" name="is_recurring" value="1"
                                   {{ old('is_recurring', $schedule->is_recurring ?? false) ? 'checked' : '' }}>
                            <label class="form-check-label" for="is_recurring">
                                Evento recorrente
                            </label>
                        </div>
                    </div>

                    <div id="recurrence_options" style="display: {{ old('is_recurring', $schedule->is_recurring ?? false) ? 'block' : 'none' }};">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="recurrence_pattern" class="form-label">Padrão de Recorrência</label>
                                    <select class="form-select" id="recurrence_pattern" name="recurrence_pattern">
                                        <option value="">Selecione...</option>
                                        <option value="daily" {{ old('recurrence_pattern', $schedule->recurrence_pattern ?? '') == 'daily' ? 'selected' : '' }}>Diário</option>
                                        <option value="weekly" {{ old('recurrence_pattern', $schedule->recurrence_pattern ?? '') == 'weekly' ? 'selected' : '' }}>Semanal</option>
                                        <option value="monthly" {{ old('recurrence_pattern', $schedule->recurrence_pattern ?? '') == 'monthly' ? 'selected' : '' }}>Mensal</option>
                                        <option value="yearly" {{ old('recurrence_pattern', $schedule->recurrence_pattern ?? '') == 'yearly' ? 'selected' : '' }}>Anual</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="recurrence_end" class="form-label">Data Final da Recorrência</label>
                                    <input type="date" class="form-control" id="recurrence_end" name="recurrence_end" 
                                           value="{{ old('recurrence_end', isset($schedule) && $schedule->recurrence_end ? $schedule->recurrence_end->format('Y-m-d') : '') }}">
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="{{ route('schedules.index') }}" class="btn btn-secondary">
                            <i class="bi bi-arrow-left"></i> Voltar
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check"></i> 
                            {{ isset($schedule) ? 'Atualizar' : 'Criar' }} Evento
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// Controle de dia inteiro
document.getElementById('all_day').addEventListener('change', function() {
    const endAtField = document.getElementById('end_at');
    if (this.checked) {
        endAtField.disabled = true;
        endAtField.value = '';
    } else {
        endAtField.disabled = false;
    }
});

// Controle de recorrência
document.getElementById('is_recurring').addEventListener('change', function() {
    const recurrenceOptions = document.getElementById('recurrence_options');
    recurrenceOptions.style.display = this.checked ? 'block' : 'none';
});

// Seleção de cor
function selectColor(color) {
    document.querySelectorAll('.color-option').forEach(el => el.classList.remove('selected'));
    event.target.classList.add('selected');
    document.getElementById('color').value = color;
}

// Gerenciamento de lembretes
let reminders = [];

@if(isset($schedule) && $schedule->reminders)
    reminders = {!! json_encode($schedule->reminders) !!};
@endif

function addReminder() {
    const input = document.getElementById('reminder_input');
    const minutes = parseInt(input.value);
    
    if (minutes && minutes > 0 && !reminders.includes(minutes)) {
        reminders.push(minutes);
        updateReminderTags();
        updateReminderInput();
        input.value = '';
    }
}

function removeReminder(element, minutes) {
    reminders = reminders.filter(r => r !== minutes);
    updateReminderTags();
    updateReminderInput();
}

function updateReminderTags() {
    const container = document.getElementById('reminder_tags');
    container.innerHTML = '';
    
    reminders.sort((a, b) => a - b).forEach(minutes => {
        const tag = document.createElement('span');
        tag.className = 'reminder-tag';
        tag.innerHTML = `${minutes} min <span class="remove" onclick="removeReminder(this, ${minutes})">&times;</span>`;
        container.appendChild(tag);
    });
}

function updateReminderInput() {
    document.getElementById('reminders_string').value = reminders.join(',');
}

// Permitir adicionar lembrete com Enter
document.getElementById('reminder_input').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault();
        addReminder();
    }
});
</script>
@endpush
